// ===================================================
// Fichier mainwindow.cpp
// Application JOTO
// Copyright (c) AMEA, 2015
// Auteur : AMEA / voir joto-transfert.fr/contact.html
// Diffusé sous license LGPL
// ===================================================

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include "customizestationdialog.h"
#include "preferencesdialog.h"
#include "downloadfiledialog.h"
#include "connecttodistantstationdialog.h"
#include "logdialog.h"
#include "getlicensedialog.h"
#include "upgradesoftwaredialog.h"
#include "downloadprogressdialog.h"
#include "aboutdialog.h"

#include "global.h"
#include "server.h"

#include <QGraphicsSceneMouseEvent>
#include <QLabel>
#include <QHostInfo>
#include <QSettings>
#include <QMenu>
#include <QClipboard>
#include <QDropEvent>
#include <QMimeData>
#include <QFileDialog>
#include <QDesktopServices>
#include <QMessageBox>
#include <QPushButton>
#include <QNetworkInterface>

#ifdef Q_OS_MAC
#include <QStyleFactory>
#endif

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

NotificationCenter notificationCenter;

MainWindow *mainWindow = NULL;

QList<UserAction *> historique;

int VERSION = 1;
const char *COMPIL_DATETIME = __DATE__ " " __TIME__;

int PORTNUMBER = 8185;
int BROADCASTPORTNUMBER = 8195;

bool autoStart = false;
bool autoOpen = true;
bool autoUpdate = true;
bool useSounds = true;
int toolBarIconsType = 2;
int clickActionType = 0;

QString homeDir = QDir::home().absolutePath();

int accessType = 0;
QStringList accessDirList = QStringList() << QDir::home().absolutePath();

QMap<QString, int> stationsContactHistory;

Server *tcpService = NULL;
UDPService *udpService = NULL;
PreferencesDialog *prefsDialog = NULL;
DownloadFileDialog *downloadFileDialog = NULL;
ConnectToDistantStationDialog *connectToDistantStation = NULL;
GetLicenseDialog *getLicenseDialog = NULL;
LogDialog *logDialog = NULL;
QMessageBox *licenceMessageDialog = NULL;

int nbFilesReceived = 0;
bool updateChecked = false;
QStringList statutMessages;

QIcon Station::pcIcon, Station::upIcon, Station::downIcon, Station::targetIcon;
QMap<QString, QPixmap *> Station::fileToIcon;

QIcon wwwIcon, licenseIcon, updatesIcon, infosIcon;

QStringList Station::filesList = QStringList() << "iMacModerne.png" <<
										 "iMacRetro.png" <<
										 "MacBookModerne.png" <<
										 "MacBookRetro.png" <<
										 "MacPro.png" <<
										 "PCBureauEcran1.png" << "PCBureauEcran2.png" << "PCBureauEcran3.png" <<"PCBureauEcran4.png" << "PCBureauEcran5.png" << "PCBureauEcran6.png" <<
										 "portable1.png" << "portable2.png" << "portable3.png" <<
										 "portable4.png" << "portable5.png" << "portable6.png" <<
										 "PortableHTC.png" <<
										 "iMacAlu.png" << "iMacG4.png" << "iMaciSight.png" << "macMini1.png" << "macMini2.png" <<
										 "macPro2.png" <<
										 "powerMacG4.png" << "serveur1.png" << "tour.png" << "XServeMac.png";

QStringList Station::namesList = QStringList() << tr( "iMac moderne") << tr( "iMac rétro") << tr( "MacBook moderne") << tr( "MacBook rétro")
									<< tr( "Mac Pro") << tr( "PC de bureau 1") << tr( "PC de bureau 2") << tr( "PC de bureau 3") << tr( "PC de bureau 4") << tr( "PC de bureau 5") << tr( "PC de bureau 6")
									<< tr( "Portable 1") << tr( "Portable 2") << tr( "Portable 3") << tr( "Portable 4") << tr( "Portable 5") << tr( "Portable 6")
									<< tr( "Portable HTC")
									<< tr( "iMac Alu") << tr( "iMac G4") << tr( "iMac iSight") << tr( "Mac Mini") << tr( "Mac Mini Alu")
									<< tr( "Mac Pro")
									<< tr( "Power Mac G4") << tr( "Serveur") << tr( "Tour") << tr( "XServe Mac");

void sendClipboardToStation( const QString &clipText, Station *station)
{
	if( station && !station->_isNotConnected)
		new SendClipboard(station, clipText);
}

void getClipboardFromStation( Station *station)
{
	if( station && !station->_isNotConnected)
		new GetClipboard(station);
}

void MovablePixmapItem::startAnimation(int animationDirection)
{
	_anim->setDuration( 750);
	_anim->setLoopCount(-1);
	_anim->setStartValue(animationDirection == 1 ? pos() : pos() + QPointF(0,16));
	_anim->setEndValue(animationDirection == 1 ? pos() + QPointF(0,16) : pos());
	// cubic curve
	_anim->setEasingCurve(QEasingCurve::OutCubic);
	_anim->start(QAbstractAnimation::KeepWhenStopped);
}

class PCItem : public QGraphicsPixmapItem
{
public :
	PCItem(int index, const QPixmap &pixmap) : QGraphicsPixmapItem(pixmap), _index( index) {
		setFlag( QGraphicsItem::ItemIsSelectable);
		setAcceptHoverEvents(true);
	}

protected :
	int _index;

	void hoverEnterEvent(QGraphicsSceneHoverEvent *)
	{
		if( mainWindow)
			mainWindow->machineHovered(_index);
	}

	void hoverLeaveEvent(QGraphicsSceneHoverEvent *)
	{
		if( mainWindow)
			mainWindow->machineHovered(-1);
	}

	void mousePressEvent( QGraphicsSceneMouseEvent *event)
	{
		if( mainWindow)
			mainWindow->machineClicked(_index, event);
	}
};

class TransfertButtonItem : public QGraphicsPixmapItem
{
public :
	TransfertButtonItem(QPointF cursorPos, QGraphicsItem *parent) : QGraphicsPixmapItem(parent),
		im1(QIcon("://icones/bouton_envoi_recoit_1.png").pixmap(128, 128)),
		im2(QIcon("://icones/bouton_envoi_recoit_2.png").pixmap(128, 128))
	{
		int x = mapToParent(cursorPos).x();
		setPixmap( x < 64 ? im1 : im2);
		setAcceptHoverEvents(true);
		setPos(0,32);
	}

protected :
	QPixmap im1, im2;

	void hoverMoveEvent(QGraphicsSceneHoverEvent *event)
	{
		int x = mapToParent(event->pos()).x();
		setPixmap( x < 64 ? im1 : im2);
		setPos(0,32);
	}
};


AutoConnectStation::AutoConnectStation(QString _stationAddress) : stationAddress( _stationAddress)
{
	qDebug() << "Test Connect station " << stationAddress;
	connect( this, &AutoConnectStation::finished, this, &AutoConnectStation::deleteLater);
	start();
}

void AutoConnectStation::run()
{
	ExtendedSocket socket;
	try
	{
		socket.connectToHost(stationAddress, PORTNUMBER);
		if( socket.waitForConnectedWithTimer())
		{
			qDebug() << "Test Connect station " << stationAddress << "OK ";
			stationsContactHistory[stationAddress] = 0;
			Station::sendInfosTo(socket);
			socket.closeConnection();
		}
		else {
			throw socket.errorString();
		}
	}
	catch( QString msg)
	{
		qDebug() << "Test Connect station " << stationAddress << "FAILED " << msg;
		stationsContactHistory[stationAddress] = stationsContactHistory[stationAddress] + 1;
		socket.abort();
	}
}


bool UpdaterHandler::isRunning = false;

UpdaterHandler::UpdaterHandler(QObject *parent)
	: QObject(parent)
	, manager(this)
	, url( UPDATEURL "JOTO.txt")
	, request(this->url)
	, indexFile( 0)
	, lastversion( 0)
	, downloadProgressDialog( NULL)
	, isAborted( false)
{
}

UpdaterHandler::~UpdaterHandler()
{
	isRunning = false;
	qDebug() << "Fin de la mise à jour";
}

void UpdaterHandler::launch(QString _label, bool _forceAutoUpdate, bool _displayResult)
{
	statutMessages += tr("Vérification de la disponibilité d'une mise à jour...");
	isRunning = true;
	qDebug() << "UpdaterHandler : " << _forceAutoUpdate;
	label = _label;
	displayResult = _displayResult;
	forceAutoUpdate = _forceAutoUpdate;
	reply = manager.get(request);
	connect( reply, SIGNAL( finished()), this , SLOT( updateFileDownloaded()));
}

void UpdaterHandler::startRequest( const QString &urlToDownload)
{
	fileInProgress = urlToDownload;
	QString fullUrl = UPDATEURL "v" + QString::number(lastversion) + "/" + urlToDownload;
	url.setUrl(fullUrl);
	request.setUrl(url);
	qDebug() << "downloading " << url.toString();
	reply = manager.get(request);
	connect( reply, &QNetworkReply::finished, this , &UpdaterHandler::updateUpgradeFileDownloaded);
	connect( reply, &QNetworkReply::downloadProgress, this , &UpdaterHandler::progress);
	connect( reply, SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(updateUpgradeFileDownloadError(QNetworkReply::NetworkError)));
}

void UpdaterHandler::updateFileDownloaded()
{
	qDebug() << "UPDATERHANDLER updateFileDownloaded : " << reply->errorString();

	QByteArray array = reply->readAll();
	if( array.size() == 0)
	{
		QMessageBox::critical(NULL, tr("Erreur durant la recherche de nouvelle version de JOTO"),
								tr("Le fichier de description des versions est introuvable.\n\n"
								"Vérifiez si vous êtes bien connecté à Internet, et relancez la recherche de mise à jour."),
								QMessageBox::Ok);
		qDebug() << "ERREUR : fichier vide ou introuvable...";
		qDebug() << array;
		deleteLater();
		return;
	}
	QStringList lines = QString(array).split('\n');
	foreach( QString line, lines)
	{
		QStringList keyValue = line.split("=");
		qDebug() << keyValue;
		if( keyValue.length() == 2)
			attributs[ keyValue[ 0]] = keyValue[ 1];
	}
	if( attributs.isEmpty() || !attributs.contains(STATIONOS "file0URL") || !attributs.contains("lastversion"))
	{
		QMessageBox::critical(NULL, tr("Erreur durant la recherche de nouvelle version de JOTO"),
								tr("Le fichier de description des versions est non conforme.\n\n"
								"Veuillez contacter AMEA pour signaler ce problème,"
								"ou rendez-vous sur le site de JOTO pour télécharger la dernière version manuellement."),
								QMessageBox::Ok);
		qDebug() << "ERREUR : fichier non conforme...";
		qDebug() << attributs;
		deleteLater();
		return;
	}
	lastversion = attributs["lastversion"].toInt();
	if( lastversion > VERSION)
	{
		disconnect( reply, SIGNAL( finished()), this , SLOT( updateFileDownloaded()));

		if( forceAutoUpdate || UpgradeSoftwareDialog::AskForUpdate( attributs["description"]) > 0)
		{
			foreach( Station *s, Station::_machines)
			{
				if( !s->_isNotConnected && s->_stationVersion == lastversion && s->_stationOS == QString( STATIONOS))
				{
					try
					{
						ExtendedSocket _socket;
						_socket.connectToHost(s->_address.toString(), PORTNUMBER);
						if( _socket.waitForConnectedWithTimer())
						{
							Station::sendInfosTo(_socket);
							_socket.sendLine( "sendmeupdate()");
							_socket.closeConnection();
						}
					}
					catch(QString msg)
					{
						statutMessages += tr( "Erreur réseau durant le transfert (demande de mise à jour auto) : %1").arg(msg);
					}
					QMessageBox::information(NULL, tr("Nouvelle version de JOTO en cours de téléchargement."),
										  label + "\n" + tr("La nouvelle version sera disponible dans votre dossier Réception"),
										  QMessageBox::Ok);
					deleteLater();
					return;
				}
			}

			downloadProgressDialog = new DownloadProgressDialog();
			downloadProgressDialog->show();
			connect( downloadProgressDialog, &DownloadProgressDialog::rejected, this, &UpdaterHandler::downloadCanceled);
			startRequest( attributs[STATIONOS "file0URL"]);
			if( !forceAutoUpdate && UpgradeSoftwareDialog::returnValue == 2)
			{
				foreach( Station *s, Station::_machines)
				{
					if( s->_isNotConnected)
						continue;
					try
					{
						ExtendedSocket _socket;
						_socket.connectToHost(s->_address.toString(), PORTNUMBER);
						if( _socket.waitForConnectedWithTimer())
						{
							Station::sendInfosTo(_socket);
							_socket.sendLine( "upgradeauto()");
							_socket.closeConnection();
						}
					}
					catch(QString msg)
					{
						statutMessages += tr( "Erreur réseau durant le transfert (demande de mise à jour auto) : %1").arg(msg);
					}
				}
			}
		}
		else
			deleteLater();
	}
	else
	{
		statutMessages += tr("Votre version de JOTO est à jour.");
		if( !forceAutoUpdate && displayResult)
			QMessageBox::information(NULL, tr("JOTO est à jour."),
							  tr("Vous avez déjà la dernière version de JOTO."),
							  QMessageBox::Ok);
		deleteLater();
	}
}

void UpdaterHandler::progress(qint64 bytesReceived, qint64 bytesTotal)
{
	downloadProgressDialog->setInfos(fileInProgress, bytesTotal > 0 ?  100 * bytesReceived / bytesTotal : 100);
}

void UpdaterHandler::updateUpgradeFileDownloaded()
{
	if( isAborted)
		return;

	QString destFileName = attributs[ QString(STATIONOS "file%1URL").arg(indexFile)];
	QDir home( QDir::homePath());
	if( home.exists("reception") == false)
		home.mkdir("reception");

	QFileInfo fileInfo( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( destFileName));
	QDir parentDir( fileInfo.path());
	if( parentDir.exists() == false)
	{
		dirsCreated.append( parentDir);
		parentDir.mkpath(".");
	}

	QFile file( fileInfo.absoluteFilePath(), this);
	file.open(QIODevice::WriteOnly);
	file.write(reply->readAll());
	file.close();

	indexFile++;
	QString newAttrName = QString(STATIONOS "file%1URL").arg(indexFile);
	if( attributs.contains( newAttrName))
	{
		startRequest( attributs[newAttrName]);
	}
	else
	{
#ifdef Q_OS_MAC
		if( attributs.contains("mac:permissionEXE")) {
			QString exeName = attributs["mac:permissionEXE"];
			QFile f(QString("%1/%2").arg(home.absoluteFilePath("reception")).arg( exeName));
			if( f.exists())
				f.setPermissions(f.permissions() | QFile::Permission(QFileDevice::ExeGroup) | QFile::Permission(QFileDevice::ExeOther) | QFile::Permission(QFileDevice::ExeOwner) | QFile::Permission(QFileDevice::ExeUser ));
		}
#endif
		if( downloadProgressDialog)
			downloadProgressDialog->accept();
		delete downloadProgressDialog;
		if( QMessageBox::question(NULL, tr("Nouvelle version de JOTO téléchargée."),
							  label + "\n" + tr("La nouvelle version est disponible dans votre dossier Réception,\nvoulez-vous ouvrir les répertoires concernés ?"),
							  QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
			openReceiveBox();
			openApplicationDir();
		}
		deleteLater();
	}
}

void UpdaterHandler::updateUpgradeFileDownloadError(QNetworkReply::NetworkError error)
{
	qDebug() << "Erreur reseau UpdaterHandler : " << error;
	if( error == 5)
		isAborted = true;
}

void UpdaterHandler::downloadCanceled()
{
	reply->abort();
	isAborted = true;
	// effacer les fichiers déjà téléchargés
	QDir home( QDir::homePath());
	for( int i = 0; i <= indexFile; i++)
	{
		QString destFileName = attributs[QString(STATIONOS "file%1URL").arg(i)];
		QFileInfo fileInfo( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( destFileName));
		qDebug() << "Effacement de " << fileInfo.absoluteFilePath();
		if( fileInfo.exists())
			fileInfo.dir().remove(fileInfo.fileName());
	}
	// effacer les dossiers créés à l'occasion du téléchargement
	foreach(QDir d, dirsCreated)
		d.rmpath(".");
	QMessageBox::information(NULL, tr("Annulation."), tr("Le téléchargement a été annulé."), QMessageBox::Ok);
	deleteLater();
}


AskLicense::AskLicense(Station *_station) : station( _station)
{
	qDebug() << "Ask license to station " << station->getDisplayName();
	connect( this, &AskLicense::finished, this, &AskLicense::deleteLater);
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &AskLicense::abort, Qt::DirectConnection);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &AskLicense::stationDeleted, Qt::DirectConnection);
	start();
}

void AskLicense::run()
{
	ExtendedSocket socket;
	try
	{
		socket.connectToHost(station->_address.toString(), PORTNUMBER);
		if( socket.waitForConnectedWithTimer() && station)
		{
			QString key, value;
			QDateTime licenseDate, firstRunDate;
			qDebug() << "Ask license to " << station->getDisplayName() << "OK ";
			Station::sendInfosTo(socket);
			socket.sendLine( "asklicense()");
			qDebug() << socket.state();

			key = socket.readLine();
			value = socket.readLine();
			licenseDate = QDateTime::fromString(socket.readLine());
			firstRunDate = QDateTime::fromString(socket.readLine());
			socket.closeConnection();

			if( firstRunDate.isValid() && ( !LicenseManager::getFirstRunDate().isValid() || firstRunDate < LicenseManager::getFirstRunDate()))
				LicenseManager::setFirstRunDate(firstRunDate);
			if( !key.isEmpty() && !value.isEmpty())
			{
				if( LicenseManager::getLicenseLevel() < 2 || LicenseManager::getDate() < licenseDate)
				{
					int prevLicenceLevel = LicenseManager::getLicenseLevel();
					LicenseManager::setKeyAndValue(key, value);
					qDebug() << "updated license with " << key << ", " << value << "(" << licenseDate.toString() << ") received from " << station->getDisplayName();
					if( LicenseManager::getLicenseLevel() > prevLicenceLevel)
						emit mainWindow->closeLicenceMessageDialog();
				}
			}
			if( mainWindow)
				emit mainWindow->refreshScene();
		}
		else
		{
			qDebug() << "Erreur AskLicense::onConnected : TIMEOUT ";
		}
	}
	catch( QString msg)
	{
		qDebug() << "Ask license to station " << (station ? station->getDisplayName() : "NULLNAME") << "FAILED " << msg;
	}
}

void AskLicense::stationDeleted( Station *stationDeleted)
{
	if( station == stationDeleted)
		station = NULL;
}

void AskLicense::abort()
{
	station = NULL;
}




SendClipboard::SendClipboard(Station *_station, QString _clipText) : station( _station), clipText(_clipText)
{
	qDebug() << "send clipboard to station " << station->getDisplayName();
	connect( this, &SendClipboard::finished, this, &SendClipboard::deleteLater);
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &SendClipboard::abort, Qt::DirectConnection);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &SendClipboard::stationDeleted, Qt::DirectConnection);
	start();
}

void SendClipboard::run()
{
	ExtendedSocket socket;
	try
	{
		socket.connectToHost(station->_address.toString(), PORTNUMBER);
		if( socket.waitForConnectedWithTimer() && station)
		{
			Station::sendInfosTo(socket);
			socket.sendLine("setclipboard");
			socket.sendLine( clipText.toUtf8());
			socket.closeConnection();
			statutMessages += tr("Envoi du presse-papier OK.");
		}
		else
			statutMessages += tr("Timeout réseau durant l'envoi du presse-papier.");
	}
	catch( QString msg)
	{
		statutMessages += tr("Erreur durant l'envoi du presse-papier vers %1 : %2").arg(station->getDisplayName()).arg(msg);
	}
}

void SendClipboard::stationDeleted( Station *stationDeleted)
{
	if( station == stationDeleted)
		abort();
}

void SendClipboard::abort()
{
	station = NULL;
}


GetClipboard::GetClipboard(Station *_station) : station( _station)
{
	qDebug() << "ask clipboard to station " << station->getDisplayName();
	connect( this, &GetClipboard::finished, this, &GetClipboard::deleteLater);
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &GetClipboard::abort, Qt::DirectConnection);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &GetClipboard::stationDeleted, Qt::DirectConnection);
	start();
}

void GetClipboard::run()
{
	ExtendedSocket socket;
	try
	{
		socket.connectToHost(station->_address.toString(), PORTNUMBER);
		if( socket.waitForConnectedWithTimer() && station)
		{
			Station::sendInfosTo(socket);
			socket.sendLine("getclipboard");
			QString value = socket.readLine();
			socket.closeConnection();
			statutMessages += tr("Demande du presse-papier OK.");
			QApplication::clipboard()->setText(value);
		}
		else
			statutMessages += tr("Timeout réseau durant la demande du presse-papier.");
	}
	catch( QString msg)
	{
		statutMessages += tr("Erreur durant la demande du presse-papier vers %1 : %2").arg(station->getDisplayName()).arg(msg);
	}
}

void GetClipboard::stationDeleted( Station *stationDeleted)
{
	if( station == stationDeleted)
		abort();
}

void GetClipboard::abort()
{
	station = NULL;
}


MainWindow::MainWindow(QWidget *parent) :
	QMainWindow(parent),
	ui(new Ui::MainWindow),
	indexSelection( -1),
	hoveredMachine( -1),
	dragInProgress( false),
	clipboardMenu(NULL),
	progressLabelWidget(new QLabel())
{
	progressLabelWidget->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
	new Station(QHostInfo::localHostName(), QHostAddress(QHostAddress::LocalHost), true);
	ui->setupUi(this);
	ui->statusBar->addWidget(progressLabelWidget, 1);
#ifdef Q_OS_MAC
	foreach( QAction *a, ui->mainToolBar->actions()) {
		foreach( QWidget *w, a->associatedWidgets())
			w->setStyle(QStyleFactory::create("windows"));
	}
#endif
	setFont(QFont(FONTNAMETOUSE));

	Station::pcIcon.addFile(QStringLiteral(":/icones/laptop.png"), QSize(), QIcon::Normal, QIcon::Off);
	Station::upIcon.addFile(QStringLiteral(":/icones/arrow_up.png"), QSize(), QIcon::Normal, QIcon::Off);
	Station::downIcon.addFile(QStringLiteral(":/icones/arrow_down.png"), QSize(), QIcon::Normal, QIcon::Off);
	Station::targetIcon.addFile(QStringLiteral(":/icones/target.png"), QSize(), QIcon::Normal, QIcon::Off);

	wwwIcon.addFile(QStringLiteral(":/icones/www.png"), QSize(), QIcon::Normal, QIcon::Off);
	licenseIcon.addFile(QStringLiteral(":/icones/license.png"), QSize(), QIcon::Normal, QIcon::Off);
	updatesIcon.addFile(QStringLiteral(":/icones/updates.png"), QSize(), QIcon::Normal, QIcon::Off);
	infosIcon.addFile(QStringLiteral(":/icones/information.png"), QSize(), QIcon::Normal, QIcon::Off);

	ui->graphicsView->setScene( &scene);
	scene.setBackgroundBrush( QColor( 128, 128, 128));

	buildScene();

	this->setAcceptDrops(true);
	ui->graphicsView->setAcceptDrops(false);

	QSettings qsettings( "AMEA", "JOTO" );

	autoStart = qsettings.value("autoStart", autoStart).toBool();
	PORTNUMBER = qsettings.value("TCPportNumber", PORTNUMBER).toInt();
	BROADCASTPORTNUMBER = qsettings.value("UDPportNumber", BROADCASTPORTNUMBER).toInt();
	autoOpen = qsettings.value("autoOpen", autoOpen).toBool();
	autoUpdate = qsettings.value("autoUpdate", autoUpdate).toBool();
	useSounds = qsettings.value("useSounds", useSounds).toBool();
	toolBarIconsType = qsettings.value("toolBarIconsType", toolBarIconsType).toInt();
	clickActionType = qsettings.value("clickActionType", clickActionType).toInt();

	QStringList stationsContactHistoryList = qsettings.value("stationsContactHistory", QVariant()).toStringList();
	for( int i = 0; i < stationsContactHistoryList.count(); i += 2)
		stationsContactHistory[ stationsContactHistoryList[ i] ] = stationsContactHistoryList[ i + 1].toInt();

	QString usage = qsettings.value("stationUsageName", "").toString();
	int iconeIndex = qsettings.value("iconeIndex", -1).toInt();
	Station::_localStation->setUsageNameAndDispath(usage, iconeIndex);

	updateToolBar();

	homeDir = qsettings.value("homeDir", homeDir).toString();
	accessType = qsettings.value("accessType", accessType).toInt();
	accessDirList = qsettings.value("accessDirList", accessDirList).toStringList();

	LicenseManager::setFirstRunDate( qsettings.value("firstRunDate", QDateTime::currentDateTime()).toDateTime());
	mainWindow = this;
	connect( this, &MainWindow::refreshScene, this, &MainWindow::buildScene);

	connect( (QApplication *)QApplication::instance(), &QApplication::commitDataRequest, this, &MainWindow::on_sessionLogout);

	connect( &notificationCenter, &NotificationCenter::onStationAdded, this, &MainWindow::buildScene);

	connect( &notificationCenter, &NotificationCenter::onStationAdded, this, &MainWindow::askForLicenseIfNeeded);
	connect( &notificationCenter, &NotificationCenter::onStationEdited, this, &MainWindow::buildScene);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &MainWindow::buildScene);
	connect( &notificationCenter, &NotificationCenter::onStationVisibilityChanged, this, &MainWindow::buildScene);

	connect(this, SIGNAL(continueQuestionSignal(QString)), this, SLOT(continueQuestionSlot(QString)), Qt::BlockingQueuedConnection);
	connect(this, SIGNAL(changePortsSignal(int,int)), this, SLOT(changePortsSlot(int,int)));
	connect(this, &MainWindow::startAutoUpdateSignal, this, &MainWindow::startAutoUpdateSlot);

	move(qsettings.value("mainWindowpos", QPoint(200, 200)).toPoint());
	restoreState(qsettings.value("mainWindowState").toByteArray());
	resize(qsettings.value("mainWindowsize", QSize(400, 150)).toSize());

	tcpService = new Server();
	udpService = new UDPService();

	CheckStartup(autoStart);
/*	if( useSounds) {
		MyPlaySound(":/sons/son3.wav");
	}*/
	QMutableMapIterator<QString,int> iterator( stationsContactHistory);
	while( iterator.hasNext())
	{
		iterator.next();
		if( iterator.key() == Station::_localStation->_address.toString() || iterator.key().indexOf("ffff") >= 0) {
			iterator.remove();
		}
		else
			new AutoConnectStation(iterator.key());
	}

	statutMessages += tr("JOTO - Transfert de fichiers version 1.%1 (%2)").arg(VERSION, 2, 10, QLatin1Char('0')).arg(COMPIL_DATETIME);
	initNetwork();

	refreshTimer = new QTimer();
	refreshTimer->setInterval( 250);
	refreshTimer->start();
	connect( refreshTimer, &QTimer::timeout, this, &MainWindow::refreshProgress);

	logDialog = new LogDialog(NULL);
	logDialog->setVisible(qsettings.value("showLogs", false).toBool());
	logDialog->restoreGeometry(qsettings.value("logsgeom", logDialog->geometry()).toByteArray());

	if( LicenseManager::getLicenseLevel() <= 1) // période d'évaluation en cours ou finie ?
	{
		QDate endDate = LicenseManager::getFirstRunDate().date().addDays(30);
		int daysLeft = QDate::currentDate().daysTo(endDate);
		licenceMessageDialog = new QMessageBox();
		if( daysLeft < 0)
			licenceMessageDialog->setText( tr("Votre période d'évaluation est terminée depuis le %1,").arg(endDate.toString("d MMM yyyy")));
		else
			licenceMessageDialog->setText( tr("Vous êtes en période d'évaluation, elle se termine le %1 (%2 jours restants),").arg(endDate.toString("d MMM yyyy")).arg(daysLeft));
		licenceMessageDialog->setInformativeText( tr("voulez-vous acheter une licence ?"));
		QPushButton *declineButton = licenceMessageDialog->addButton( daysLeft < 0 ? tr("Non merci, pas encore.") : tr("Continuer l'essai."), QMessageBox::ActionRole);
		connect( declineButton, &QPushButton::clicked, this, &MainWindow::closeLicenceMessageDialog);
		QPushButton *buyButton = licenceMessageDialog->addButton( tr("Acheter la licence..."), QMessageBox::ActionRole);
		connect( buyButton, &QPushButton::clicked, this, &MainWindow::gotoLicenseUrl);
		licenceMessageDialog->setIcon(QMessageBox::Question);
		licenceMessageDialog->show();
	}
	QTimer::singleShot(1, this, SLOT(redrawScenes()));
}

MainWindow::~MainWindow()
{
	delete ui;
}

void MainWindow::closeLicenceMessageDialog()
{
	if( licenceMessageDialog)
		emit licenceMessageDialog->reject();
	licenceMessageDialog = NULL;
}

void MainWindow::gotoLicenseUrl()
{
	if( QDesktopServices::openUrl(QUrl(LICENSEURL)) == false) {
		QMessageBox::critical(NULL, QObject::tr("JOTO"), QObject::tr("Erreur lors du renvoi sur le site.\n\nVous pouvez y accéder à cette adresse : %1.").arg(LICENSEURL));
	}
}

void MainWindow::continueQuestionSlot(const QString msg )
{
	QMessageBox msgBox;
	msgBox.setText( tr("Erreur réseau durant le transfert : %1").arg(msg));
	msgBox.setInformativeText( tr("Que voulez-vous faire ?"));
	QPushButton *continueButton = msgBox.addButton(tr("Continuer"), QMessageBox::ActionRole);
	QPushButton *cancelButton = msgBox.addButton(tr("Tout annuler"), QMessageBox::ActionRole);
	QPushButton *stopButton = msgBox.addButton(tr("Arreter"), QMessageBox::ActionRole);
	msgBox.setIcon(QMessageBox::Question);
	msgBox.exec();
	QPushButton *a = (QPushButton *)msgBox.clickedButton();
	if( a == continueButton)
		answerButton = 1;
	else if( a == cancelButton)
		answerButton = 2;
	else if( a == stopButton)
		answerButton = 3;
}

void MainWindow::changePortsSlot(int tcpPort, int udpPort)
{
	udpService->stopService();
	delete udpService;
	// restart with new ports
	PORTNUMBER = tcpPort;
	BROADCASTPORTNUMBER = udpPort;
	udpService = new UDPService();

	tcpService->restart();
	QSettings qsettings( "AMEA", "JOTO" );

	qsettings.setValue( "TCPportNumber", PORTNUMBER);
	qsettings.setValue( "UDPportNumber", BROADCASTPORTNUMBER);
	if( prefsDialog)
		prefsDialog->updateValues();
}

void MainWindow::startAutoUpdateSlot(QString label, bool forceUpdate, bool displayResult)
{
	if( UpdaterHandler::isRunning)
		return;
	UpdaterHandler *updateHandler = new UpdaterHandler();
	updateHandler->launch(label, forceUpdate, displayResult);
}

void MainWindow::initNetwork()
{
	QString adressesTxt;
	int count = 0;
	foreach (const QHostAddress &address, QNetworkInterface::allAddresses())
	{
		if (address.protocol() == QAbstractSocket::IPv4Protocol && address != QHostAddress(QHostAddress::LocalHost))
		{
			count++;
			adressesTxt += address.toString() + "<br>";
		}
	}
	if( count == 0) {
		QTimer::singleShot(1000, this, SLOT(initNetwork()));
		return;
	}
	else if( count == 1)
		adressesTxt = tr("Adresse IP locale de la machine : <br>") + adressesTxt;
	else
		adressesTxt = tr("Adresses IP locales de la machine : <br>") + adressesTxt;
	statutMessages += adressesTxt.split("<br>");

	if( autoUpdate) {
		QTimer::singleShot(10000, this, SLOT(startAutoUpdateAtStartupSlot()));
	}
}

void MainWindow::refreshProgress()
{
	static int prev_nbReceivedFiles = 0;

	QString status = "";
	foreach( Station *s, Station::_machines)
	{
		bool headerTxt = true;
		foreach( TransfertFile *task, s->_receivingTasks)
		{
			if( !headerTxt)
				status.append(" - ");
			else
				status.append(tr("Envoi de "));
			headerTxt = false;
			if( task->_filesList.size() == 1) {
				QFileInfo f(task->_filesList.first());
				status.append( f.fileName());
			}
			else
				status.append( tr("%1 fichiers").arg( task->_filesList.size()));
			status.append( tr(" ( %1 %)").arg( task->getPourcent()));
		}

		headerTxt = true;
		foreach( TransfertFileBase *task, s->_sendingTasks)
		{
			if( !headerTxt)
				status.append(" - ");
			else
			{
				if( !status.isEmpty())
					status.append( " - ");
				status.append(tr("Réception de "));
			}
			headerTxt = false;
			if( task->_filesList.size() == 1) {
				QFileInfo f(task->_filesList.first());
				status.append( f.fileName());
			}
			else
				status.append( tr("%1 fichiers").arg( task->_filesList.size()));
			status.append( tr(" ( %1 %)").arg( task->getPourcent()));
		}
	}

	progressLabelWidget->setText(status);
	if( !statutMessages.isEmpty())
		ui->statusBar->showMessage(statutMessages.last(), 3000);
	while( !statutMessages.isEmpty())
	{
		qDebug() << "display log " << statutMessages.first();
		if( logDialog)
			logDialog->addMessage( statutMessages.first());
		statutMessages.removeFirst();
	}

	if( prev_nbReceivedFiles != nbFilesReceived)
	{
		QFont font = ui->actionReception->font();
		if( nbFilesReceived > 0) {
			ui->actionReception->setText( tr("Réception (%1)").arg(nbFilesReceived));
		}
		else {
			ui->actionReception->setText( tr("Réception"));
		}
		font.setBold( nbFilesReceived > 0);
		ui->actionReception->setFont(font);

		prev_nbReceivedFiles = nbFilesReceived;
	}
}


QMenu *MainWindow::buildContextMenu()
{
	QMenu *menu = new QMenu();
	foreach( Station *s, Station::_machines) {
		if( s->_isNotConnected)
			continue;
		QAction *action = new QAction( Station::pcIcon, tr("Vers %1").arg( s->getDisplayName()), menu);
		action->setData( s->_name);
		menu->addAction( action);
	}
	foreach( Station *s, Station::_machines) {
		if( s->_isNotConnected)
			continue;
		QAction *action = new QAction( Station::pcIcon, tr("Depuis %1").arg( s->getDisplayName()), menu);
		action->setData( s->_name + "_");
		menu->addAction( action);
	}
	if( menu->actions().isEmpty())
	{
		QAction *noMachine = new QAction( tr("Pas de machine visible..."), menu);
		noMachine->setEnabled(false);
		menu->addAction(noMachine);
	}
	return menu;
}

void MainWindow::sendClipboard(QAction *action )
{
	if( action)
	{
		QString stationName = action->data().toString().toUpper();
		if( stationName.right(1) == "_") {
			stationName = stationName.left(stationName.length() - 1);
			qDebug() << "Demande du clipboard à la machine" << stationName;

			Station *station = Station::findStation(stationName);
			if( station) {
				if( LicenseManager::checkForLicenseLevel(1))
				{
					UserAction *newAction = new GetClipboardAction(station->_name, station->getDisplayName(), QStringList());
					historique.append( newAction);
					getClipboardFromStation(station);
				}
			}
			else {
				qDebug() << "Erreur :" << stationName << "inconnue";
				statutMessages += tr( "Erreur : station %1 inconnue").arg(stationName);
			}
			return;
		}
		QString clipText = QApplication::clipboard()->text();
		if( !clipText.isEmpty())
		{
			qDebug() << "Envoi :" << clipText << "à la machine" << stationName;

			Station *station = Station::findStation(stationName);
			if( station) {
				if( LicenseManager::checkForLicenseLevel(1))
				{
					UserAction *newAction = new SendClipboardAction(station->_name, station->getDisplayName(), QStringList() << clipText);
					historique.append( newAction);
					sendClipboardToStation(clipText, station);
				}
			}
			else {
				qDebug() << "Erreur :" << stationName << "inconnue";
				statutMessages += tr( "Erreur : station %1 inconnue").arg(stationName);
			}
		}
		else {
			statutMessages += tr("Presse-papier vide ou ne contenant pas de texte.");
		}
	}
}

void MainWindow::sendFiles(const QStringList &pathList, int destinationIndex)
{
	Station *station = Station::_machines.at( destinationIndex);
	if( station) {
		if( LicenseManager::checkForLicenseLevel(1))
		{
			qDebug() << "Fichiers :" << pathList << "vers machine" << destinationIndex;

			UserAction *newAction = new SendFileAction( station->_name, station->getDisplayName(), pathList);
			historique.append( newAction);

			station->addTransfertTask( pathList);
		}
	}
}

void MainWindow::dragLeaveEvent(QDragLeaveEvent * /*event*/)
{
	if( indexSelection != -1)
	{
		indexSelection = -1;
		buildScene();
	}
}

void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
	if (event->mimeData()->hasUrls() && Station::_machines.size() > 0)
	{
		dragInProgress = true;
		event->acceptProposedAction();
	}
}

void MainWindow::dropEvent(QDropEvent *event)
{
	dragInProgress = false;
	if (indexSelection >= 0 && event->mimeData()->hasUrls()) {
		QStringList pathList;
		foreach (const QUrl &url, event->mimeData()->urls()) {
			QString f = url.toLocalFile();
#ifdef Q_OS_MAC
			if( f.length() >= 2 && f.right(1) == "/")
				f = f.left( f.length() - 1);
#endif
			pathList.append(f);
		}
		sendFiles( pathList, indexSelection);
	}
	 event->acceptProposedAction();
	 indexSelection = -1;
	 buildScene();
}

void MainWindow::dragMoveEvent(QDragMoveEvent *event)
{
	int oldSelection = indexSelection;
	indexSelection = -1;
	if (event->mimeData()->hasUrls() && Station::_machines.size() > 0)
	{
		QRect rect = event->answerRect();
		// décaler sous la barre d'outils
		rect.moveTop( rect.y() - ui->mainToolBar->geometry().height());
		if( rect.intersects(ui->graphicsView->geometry()))
		{
			event->acceptProposedAction();
			QPoint dropPos = event->pos();
			indexSelection = dropPos.x() * Station::_machines.count() / ui->graphicsView->width();
		}
	}
	if( oldSelection != indexSelection)
		buildScene();
}

void MainWindow::machineHovered(int index)
{
	if( clickActionType == 1 && hoveredMachine != index)
	{
		hoveredMachine = index;
		buildScene();
	}
}

void MainWindow::machineClicked(int index, QGraphicsSceneMouseEvent *event)
{
	indexSelection = index;
	hoveredMachine = -1;
	buildScene();

	Station *s = Station::_machines.at( indexSelection);
	if( s == NULL)
		return;

	bool doDownload = false;
	if( clickActionType == 0)
	{
		QMenu *menu = new QMenu();
		QAction *action1Menu = new QAction( Station::upIcon, tr("Envoyer..."), menu);
		action1Menu->setData( QVariant( 0));
		menu->addAction( action1Menu);
		QAction *action2Menu = new QAction( Station::downIcon, tr("Télécharger..."), menu);
		action2Menu->setData( QVariant( 1));
		menu->addAction( action2Menu);

		ui->mainToolBar->setEnabled(false);
		QAction *ac = menu->exec(QCursor::pos());
		ui->mainToolBar->setEnabled(true);
		if( ac) {
			int choice = ac->data().toInt();
			doDownload = choice == 1;
		}
		else
		{
			delete menu;
			resetSelection();
			return;
		}
		delete menu;
	}
	else if( clickActionType == 1) doDownload = event->pos().x() >= 64;
	else if( clickActionType == 2) doDownload = event->modifiers() && Qt::ShiftModifier;
	else if( clickActionType == 3) doDownload = !(event->modifiers() && Qt::ShiftModifier);

	if( doDownload)
	{
		if( downloadFileDialog == NULL)
			downloadFileDialog = new DownloadFileDialog(this);
		downloadFileDialog->setStationName( s->_name);
		downloadFileDialog->show();
		resetSelection();
	}
	else
	{
		QString sName = s->_name;
		resetSelection();
		if( LicenseManager::checkForLicenseLevel(1))
		{
			ui->mainToolBar->setEnabled(false);
			QStringList fileNames = QFileDialog::getOpenFileNames(this, tr("Envoyer un ou plusieurs fichiers"), "", tr("Tous les fichiers (*.*)"));
			ui->mainToolBar->setEnabled(true);
			if( !fileNames.isEmpty())
			{
				// récupérer le pointeur au dernier moment
				Station *s = Station::findStation( sName);
				if( s)
				{
					UserAction *newAction = new SendFileAction( s->_name, s->getDisplayName(), fileNames);
					historique.append( newAction);

					s->addTransfertTask( fileNames);
					resetSelection();
				}
			}
		}
	}
}

void MainWindow::resetSelection()
{
	int oldSelection = indexSelection;
	indexSelection = -1;
	if( oldSelection != indexSelection)
		buildScene();
}

QGraphicsSimpleTextItem *MainWindow::truncateText(double widthMax, QGraphicsScene &scene, QString text)
{
	QFont TEXTFONT(FONTNAMETOUSE, 10);
	QGraphicsSimpleTextItem *nameItem = scene.addSimpleText(text, TEXTFONT);
	nameItem->setScale(2.5);
	while( nameItem->boundingRect().width() > widthMax)
	{
		text.remove( text.length() - 1, 1);
		nameItem->setText( text+"...");
	}
	return nameItem;
}

void MainWindow::askForLicenseIfNeeded( Station *station)
{
	if( station && !station->_isNotConnected)
	{
		station->sendInfosToAllStations();
		new AskLicense( station);
	}
}

void MainWindow::buildScene()
{
	ui->graphicsView->setBackgroundBrush( QColor(82,82,82));
	scene.clear();
	scene.setSceneRect(QRectF());

	int index = 0;
	int posX = -(Station::_machines.count() - 1) * 160 / 2;
	foreach( Station *s, Station::_machines)
	{
		PCItem *pcItem = new PCItem( index, *s->getPixmap());
		scene.addItem(pcItem);
		pcItem->setTransformOriginPoint( pcItem->boundingRect().width() / 2, pcItem->boundingRect().height() / 2);
		pcItem->setPos( posX, 0);
		if( (indexSelection >= 0 && indexSelection != index) || s->_isNotConnected) {
			pcItem->setOpacity( 0.5);
		}
		else if( indexSelection >= 0) {
			pcItem->setScale( pcItem->scale() * 1.2);
		}
		if( dragInProgress == false && hoveredMachine >= 0 && index == hoveredMachine && s->_isNotConnected == false)
		{
			QPointF cursorPos = ui->graphicsView->mapToScene(ui->graphicsView->mapFromGlobal(QCursor::pos()));
			TransfertButtonItem *buttonItem = new TransfertButtonItem(cursorPos, pcItem);
			buttonItem->setTransformOriginPoint( buttonItem->boundingRect().width() / 2, buttonItem->boundingRect().height() / 2);
		}
		MovablePixmapItem *backPcItem = new MovablePixmapItem( Station::targetIcon.pixmap(32, 32));
		backPcItem->setTransformOriginPoint( backPcItem->boundingRect().width() / 2, backPcItem->boundingRect().height() / 2);
		backPcItem->setPos( posX + 48, 16);
		backPcItem->setScale(2.);
		backPcItem->setOpacity( (indexSelection == -1 || indexSelection != index) ? 0.0 : 1.0);
		scene.addItem( backPcItem );

		QGraphicsSimpleTextItem *nameItem = truncateText(64, scene, s->getDisplayName());
		if( s->_isNotConnected )
			nameItem->setText( nameItem->text() + tr("\n(en attente)"));
		nameItem->setBrush(QColor(240,240,240));
		nameItem->setPos( posX + 32 - nameItem->boundingRect().width() / 2, 128 + nameItem->boundingRect().height() * 0.5);

		if( !s->_receivingTasks.isEmpty())
		{
			MovablePixmapItem *upItem = new MovablePixmapItem( Station::upIcon.pixmap(32, 32));
			scene.addItem( upItem );
			upItem->setOpacity(.75);
			upItem->setPos( posX + (!s->_sendingTasks.isEmpty() ? 76 : 54), 100);
			upItem->startAnimation(0);
		}
		if( !s->_sendingTasks.isEmpty())
		{
			MovablePixmapItem *downItem = new MovablePixmapItem( Station::downIcon.pixmap(32, 32));
			scene.addItem( downItem );
			downItem->setOpacity(.75);
			downItem->setPos( posX + (!s->_receivingTasks.isEmpty() ? 20 : 54), 100);
			downItem->startAnimation(1);
		}
		index++;
		posX += 160;
	}
	if( index == 0)
	{
		QGraphicsPixmapItem *okItem = scene.addPixmap(QIcon("://icones/icone_ready.png").pixmap(379, 137));
		okItem->setTransformOriginPoint( okItem->boundingRect().width() / 2, okItem->boundingRect().height() / 2);
		okItem->setPos( -130, 10);
	}

	QRectF bounds = scene.sceneRect();
	if( bounds.width() > 0 && bounds.height() > 0)
	{
		double scale = min( ui->graphicsView->width() / bounds.width(), ui->graphicsView->height() / bounds.height()) * .9;
		ui->graphicsView->setSceneRect(0,0,128,180);
		ui->graphicsView->setTransform(QTransform::fromScale(scale, scale));
	}
}

void MainWindow::on_sessionLogout()
{
	qDebug() << "onSessionLogout";
	closeEvent(NULL);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	qDebug() << "closeEvent : check pending tasks";
	if( mainWindow == NULL)
		return;

	if( Station::pendingTasks()) {
		QMessageBox msgBox;
		msgBox.setText( tr("Il reste des transferts de fichiers en cours..."));
		msgBox.setInformativeText( tr("Que voulez-vous faire ?"));
		QPushButton *cancelButton = msgBox.addButton(tr("Ne pas quitter"), QMessageBox::ActionRole);
		msgBox.addButton(tr("Vraiment quitter"), QMessageBox::ActionRole);
		msgBox.setIcon(QMessageBox::Question);
		msgBox.exec();
		QPushButton *a = (QPushButton *)msgBox.clickedButton();
		if( a == cancelButton) {
			event->ignore();
			return;
		}
	}
	statutMessages += tr("Arret de JOTO en cours...");
	qDebug() << "closeEvent confirmed";
	udpService->stopService();
	delete udpService;
	tcpService->stop();
	delete tcpService;
	Station::stopTasks();
	qDebug() << "tasks stopped";

	emit notificationCenter.onApplicationQuit();
	// envoyer un message de fin aux autres stations
	foreach( Station *station, Station::_machines)
	{
		if( station->_isNotConnected)
			continue;
		try
		{
			qDebug() << "sending quit() to " << station->getDisplayName();
			ExtendedSocket _socket;
			_socket.connectToHost(station->_address.toString(), PORTNUMBER);
			if( _socket.waitForConnectedWithTimer())
			{
				Station::sendInfosTo(_socket);
				_socket.sendLine( "quit()");
				_socket.closeConnection();
			}
		}
		catch(QString msg)
		{
			qDebug() << "Erreur MainWindow::on_actionQuitter_triggered : " << msg;
		}
	}
	qDebug() << "quit() sent";

	QSettings qsettings( "AMEA", "JOTO" );
	qsettings.setValue("mainWindowsize", size());
	qsettings.setValue("mainWindowpos", pos());
	qsettings.setValue("mainWindowState", saveState());

	if( logDialog) {
		qsettings.setValue("showLogs", logDialog->isVisible());
		qsettings.setValue("logsgeom", logDialog->saveGeometry());
	}

	QStringList history;
	QMapIterator<QString,int> iterator( stationsContactHistory);
	while( iterator.hasNext())
	{
		iterator.next();
		if( iterator.value() < 5)
			history << iterator.key() << QString::number( iterator.value());
	}
	qsettings.setValue("stationsContactHistory", history);

	qsettings.setValue("firstRunDate", LicenseManager::getFirstRunDate());

	QMainWindow::closeEvent(event);

	disconnect((QApplication *)QApplication::instance(), &QApplication::commitDataRequest, this, &MainWindow::on_sessionLogout);
	mainWindow = NULL;
	QApplication::quit();
	qDebug() << "All done";
}


void MainWindow::resizeEvent ( QResizeEvent * /*ev*/)
{
	buildScene();
}

void MainWindow::on_actionPresse_papiers_triggered()
{
	QMenu *menu = buildContextMenu();
	ui->mainToolBar->setEnabled(false);
	QAction *ac = menu->exec(QCursor::pos());
	ui->mainToolBar->setEnabled(true);
	if( ac)
		sendClipboard(ac);
	delete menu;
}

void MainWindow::updateMachines()
{
	emit refreshScene();
}

void MainWindow::on_actionReception_triggered()
{
	openReceiveBox();
	nbFilesReceived = 0;
}

void MainWindow::on_actionHistorique_triggered()
{
	QMenu *menu = new QMenu();
	int index = 1;
	for( int i = 0; i < historique.count(); i++)
	{
		if( historique.at(i)->checkActionValidity())
		{
			QAction *actionMenu = new QAction( QString::number(index++) + ". " + historique.at(i)->_libelle, menu);
			actionMenu->setData( QVariant( i));
			menu->addAction( actionMenu);
		}
	}
	if( menu->actions().isEmpty())
	{
		QAction *noMachine = new QAction( tr("Pas encore d'action enregistrée..."), menu);
		noMachine->setEnabled(false);
		menu->addAction(noMachine);
	}
	ui->mainToolBar->setEnabled(false);
	QAction *ac = menu->exec(QCursor::pos());
	ui->mainToolBar->setEnabled(true);
	if( ac) {
		if( LicenseManager::checkForLicenseLevel(1))
		{
			UserAction *a = historique.at( ac->data().toInt());
			if( a)
				a->performAction();
		}
	}
	delete menu;
}

void MainWindow::on_actionReglages_triggered()
{
	PreferencesDialog *dialog = new PreferencesDialog(this);
	if( dialog->exec())
	{
		int newtcpport, newudpport;
		dialog->getValues( newtcpport, newudpport, autoStart, autoOpen, autoUpdate, useSounds, toolBarIconsType, clickActionType, homeDir, accessType, accessDirList);
		/* qDebug() << "Nouveaux réglages : TCP : " << newtcpport << ", udp : " << newudpport
				 << ", autoStart : " << autoStart << ", autoOpen : " << autoOpen
				 << ", autoUpdate : " << autoUpdate
				 << ", useSounds : " << useSounds << ", toolBarIconTypes : " << toolBarIconsType
				 << ", clickActionType : " << clickActionType
				 << ", homeDir : " << homeDir
				 << ", accessType : " << accessType
				 << ", accessDirList : " << accessDirList; */
		CheckStartup( autoStart);
		updateToolBar();

		if( newtcpport != PORTNUMBER || newudpport != BROADCASTPORTNUMBER)
		{
			udpService->stopService();
			delete udpService;

			// envoyer un message de mise à jour aux autres stations
			foreach( Station *station, Station::_machines)
			{
				if( station->_isNotConnected)
					continue;
				try
				{
					ExtendedSocket _socket;
					_socket.connectToHost(station->_address.toString(), PORTNUMBER);
					if( _socket.waitForConnectedWithTimer())
					{
						Station::sendInfosTo(_socket);

						_socket.sendLine( "newports");
						_socket.sendLine( QString::number( newtcpport));
						_socket.sendLine( QString::number( newudpport));
						_socket.closeConnection();
					}
				}
				catch(QString msg)
				{
					qDebug() << "Erreur MainWindow::on_actionReglages_triggered : " << msg;
				}
			}
			PORTNUMBER = newtcpport;
			BROADCASTPORTNUMBER = newudpport;
			udpService = new UDPService();
			tcpService->restart();
		}
	}
	delete dialog;
}

void MainWindow::on_actionOutils_triggered()
{
	QMenu *menu = new QMenu();
	QAction *actionMenu;

	actionMenu = new QAction( Station::pcIcon, tr( "Se connecter à une machine..."), menu);
	actionMenu->setData( QVariant( 0));
	menu->addAction( actionMenu);

	actionMenu = new QAction( wwwIcon, tr( "Aller sur le site..."), menu);
	actionMenu->setData( QVariant( 1));
	menu->addAction( actionMenu);

	actionMenu = new QAction( wwwIcon, tr( "Documentation en ligne..."), menu);
	actionMenu->setData( QVariant( 5));
	menu->addAction( actionMenu);

	actionMenu = new QAction( licenseIcon, LicenseManager::getLicenseLevel() < 2 ? tr("Entrer la licence...") : tr("Modifier la licence..."), menu);
	actionMenu->setData( QVariant( 2));
	menu->addAction( actionMenu);

	menu->addSeparator();

	actionMenu = new QAction( updatesIcon, tr( "Vérifier les mises à jour disponibles..."), menu);
	actionMenu->setData( QVariant( 3));
	menu->addAction( actionMenu);

	actionMenu = new QAction( infosIcon, tr( "A propos..."), menu);
	actionMenu->setData( QVariant( 4));
	menu->addAction( actionMenu);

	ui->mainToolBar->setEnabled(false);
	QAction *ac = menu->exec(QCursor::pos());
	ui->mainToolBar->setEnabled(true);
	if( ac)
	{
		int indexAction = ac->data().toInt();
		switch( indexAction)
		{
		case 0:
			if( connectToDistantStation == NULL)
				connectToDistantStation = new ConnectToDistantStationDialog( this);
			connectToDistantStation->show();
			break;
		case 1 :
			if( QDesktopServices::openUrl(QUrl(WEBSITEURL)) == false) {
				QMessageBox::critical(NULL, tr("JOTO"),tr("Erreur lors du renvoi sur le site.\n\nVous pouvez y accéder à cette adresse : %1.").arg(WEBSITEURL));
			}
			break;
		case 2 :
			if( getLicenseDialog == NULL)
				getLicenseDialog = new GetLicenseDialog(this);
			getLicenseDialog->show();
			break;
		case 3 :
			if( mainWindow)
				emit mainWindow->startAutoUpdateSignal("",false, true);
			break;
		case 4 :
			{
				AboutDialog about;
				about.exec();
			}
			break;
		case 5 :
			if( QDesktopServices::openUrl(QUrl(DOCSURL)) == false) {
				QMessageBox::critical(NULL, tr("JOTO"),tr("Erreur lors du renvoi sur le site.\n\nVous pouvez y accéder à cette adresse : %1.").arg(DOCSURL));
			}
			break;
		}
	}
	delete menu;
}

void MainWindow::on_actionMessages_triggered()
{
	if( logDialog)
		logDialog->setVisible( !logDialog->isVisible());
}

void MainWindow::updateToolBar()
{
	ui->mainToolBar->setToolButtonStyle( toolBarIconsType == 2 ? Qt::ToolButtonTextUnderIcon : Qt::ToolButtonIconOnly);
	ui->mainToolBar->setIconSize( toolBarIconsType == 0 ? QSize(16, 16) : QSize( 24, 24));
	ui->actionMamachine->setText(Station::_localStation->getDisplayName());
	ui->actionMamachine->setIcon(QIcon(*Station::_localStation->getPixmap()));
}

void MainWindow::on_actionMamachine_triggered()
{
	CustomizeStationDialog *dialog = new CustomizeStationDialog(this, Station::_localStation);
	if( dialog->exec()) {
		updateToolBar();
	}
	delete dialog;
}
