// ===================================================
// Fichier downloadfiledialog.cpp
// Application JOTO
// Copyright (c) AMEA, 2015
// Auteur : AMEA / voir joto-transfert.fr/contact.html
// Diffusé sous license LGPL
// ===================================================

#include "global.h"

#include "downloadfiledialog.h"
#include "ui_downloadfiledialog.h"

#include <QStandardItemModel>
#include <QDateTime>
#include <QSettings>

class FilesModel : public QStandardItemModel
{
public :
	FilesModel() : QStandardItemModel(NULL)
	{
		connect( this, &FilesModel::itemChanged, this, &FilesModel::clicked);
		setSortRole(Qt::UserRole);
	}

	void clicked(void)
	{
		if( downloadFileDialog)
			downloadFileDialog->itemClicked();
	}

	void appendRow(bool isDirectory, const QString &fileName, double fileSize, const QDateTime &fileDate)
	{
		QString fileSizeTxt = tr("<rep>");
		if( isDirectory == false)
		{
			if( fileSize > 1024 * 1024 * 1024)  fileSizeTxt = QString::number( fileSize / 1024 / 1024 / 1024, 'f', 1) + " Go";
			else if( fileSize > 1024 * 1024)    fileSizeTxt = QString::number( fileSize / 1024 / 1024, 'f', 1) + " Mo";
			else if( fileSize > 1024)           fileSizeTxt = QString::number( fileSize / 1024, 'f', 1) + " Ko";
			else                                fileSizeTxt = QString::number( fileSize);
		}

		QStandardItem *fileItem = new QStandardItem(isDirectory ? QIcon(":/icones/dir_icone.png") : QIcon(":/icones/file_icone.png"), fileName);
		fileItem->setData(isDirectory);
		fileItem->setCheckable(true);
		fileItem->setData(fileName, Qt::UserRole);

		QStandardItem *sizeItem = new QStandardItem( fileSizeTxt);
		sizeItem->setData(fileSize, Qt::UserRole);
		sizeItem->setData(Qt::AlignRight, Qt::TextAlignmentRole);

		QStandardItem *dateItem = new QStandardItem( fileDate.toString("dd/MM/yyyy HH:mm:ss"));
		dateItem->setData(fileDate, Qt::UserRole);
		dateItem->setData(Qt::AlignRight, Qt::TextAlignmentRole);

		QStandardItemModel::appendRow(QList<QStandardItem*>() << fileItem << sizeItem << dateItem);
	}

	void appendRow(const QString &volumeName)
	{
		QStandardItem *volItem = new QStandardItem(QIcon(":/icones/disk_icone.png"), volumeName);
		volItem->setData(true);
		QStandardItemModel::appendRow( QList<QStandardItem*>() << volItem);
	}
};

FilesModel filesModel;
DownloadFilesList filesListDownloader;
DownloadVolumesList volumesListDownloader;

DownloadFilesList::DownloadFilesList( )
	: QThread(), ui( 0), selectedStation(), tempSocket(NULL)
{
	connect(this, &DownloadFilesList::finished, this, &DownloadFilesList::onEndThread);
}

void DownloadFilesList::GetFilesList( Ui::DownloadFileDialog *&_ui, Station *_selectedStation)
{
	qDebug() << "GetFileList : " << isRunning();
	if( isRunning()) {
		if( tempSocket) tempSocket->abort();
		wait();
	}
	ui = _ui;
	selectedStation = _selectedStation->_name;
	if( downloadFileDialog && ui) {
		ui->lInfos->setText(tr( "Lecture du dossier distant..."));
		setButtonsStates(false);
	}
	start();
}

void DownloadFilesList::onEndThread()
{
	if( downloadFileDialog) {
		ui->lInfos->setText(messageToDisplay);
		ui->lCurDir->setText(newDirToDisplay);
		setButtonsStates(true);
	}
}

void DownloadFilesList::setButtonsStates( bool states)
{
	if( downloadFileDialog && ui) {
		ui->bUp->setEnabled(states);
		ui->bHome->setEnabled(states);
		ui->bDisks->setEnabled(states);
	}
}

void DownloadFilesList::run()
{
	try
	{
		ExtendedSocket _socket;
		Station *s = Station::findStation(selectedStation);
		if( s == NULL) return;
		qDebug() << s->getDisplayName();
		tempSocket = &_socket;
		_socket.connectToHost(s->_address.toString(), PORTNUMBER);
		if( _socket.waitForConnectedWithTimer())
		{
			messageToDisplay = "";
			filesModel.removeRows(0, filesModel.rowCount());

			bool ok = false;
			Station::sendInfosTo(_socket);
			_socket.sendLine( "listfiles");
			_socket.sendLine( s->_currentDirPath);
			s->_currentDirPath = _socket.readLine();
			newDirToDisplay = s->_currentDirPath;
			while( true)
			{
				QString cmd = _socket.readLine();
				if( cmd == QStringLiteral("end"))
					break;
				else if( cmd == QStringLiteral("message"))
				{
					messageToDisplay = _socket.readLine();
					break;
				}
				else if( cmd == QStringLiteral("file"))
				{
					QString fileName = _socket.readLine().simplified();
					QString fileSizeTxt = _socket.readLine().simplified();
					QString fileDateTxt = _socket.readLine().simplified();
					bool isDirectory = _socket.readLine().simplified().toInt(&ok);
					double size = fileSizeTxt.toDouble(&ok);
					QDateTime date = QDateTime::fromString( fileDateTxt, "dd/MM/yyyy HH:mm:ss");
					if( !fileName.isEmpty() && fileName != "." && fileName != "..")
					{
						filesModel.appendRow( isDirectory, fileName, size, date);
					}
				}
				else if( !cmd.isEmpty())
				{
					qDebug() << "Erreur : reçu " << cmd;
					Station *s = Station::findStation(selectedStation);
					messageToDisplay = "Message reçu de " + (s ? s->getDisplayName() : "<NULL>") + " : " + cmd;
					break;
				}
			}
			_socket.closeConnection();
			if( filesModel.rowCount() == 0 && messageToDisplay.isEmpty())
				messageToDisplay = tr("Pas de fichier dans ce repertoire");
		}
		else {
			Station *s = Station::findStation(selectedStation);
			messageToDisplay = tr( "Erreur : impossible de se connecter à %1.").arg(s ? s->getDisplayName() : "<NULL>");
		}
	}
	catch(QString msg)
	{
		messageToDisplay = msg;
	}
	tempSocket = NULL;
	return;
}

DownloadVolumesList::DownloadVolumesList()
	: DownloadFilesList()
{
}

void DownloadVolumesList::GetVolumesList(Ui::DownloadFileDialog *&_ui, Station *_selectedStation)
{
	if( isRunning()) {
		if( tempSocket) tempSocket->abort();
		wait();
	}
	ui = _ui;
	selectedStation = _selectedStation->_name;
	if( downloadFileDialog && ui) {
		ui->lCurDir->clear();
		ui->lInfos->setText(tr( "Demande de la liste des disques distants..."));
		setButtonsStates(false);
	}
	start();
}

void DownloadVolumesList::run()
{
	try
	{
		Station *s = Station::findStation(selectedStation);
		if( s == NULL) return;
		s->_currentDirPath = "";
		ExtendedSocket _socket;
		qDebug() << s->getDisplayName();
		tempSocket = &_socket;
		_socket.connectToHost(s->_address.toString(), PORTNUMBER);
		if( _socket.waitForConnectedWithTimer())
		{
			Station::sendInfosTo(_socket);
			_socket.sendLine( QString( "listvolumes"));
			while( true)
			{
				QString volumeName = _socket.readLine();
				if( volumeName.isEmpty() || volumeName == QStringLiteral("end"))
					break;
				filesModel.appendRow(volumeName);
			}
			_socket.closeConnection();
			messageToDisplay = "";
		}
		else
		{
			Station *s = Station::findStation(selectedStation);
			messageToDisplay = tr( "Echec de la connexion avec %1.").arg(s ? s->getDisplayName() : "<NULL>");
		}
	}
	catch(QString msg)
	{
		qDebug() << "Erreur DownloadVolumesList : " << msg;
		messageToDisplay = msg;
	}
	tempSocket = NULL;
	return;
}


DownloadFileDialog::DownloadFileDialog(QWidget *parent) :
	QDialog(parent),
	ui(new Ui::DownloadFileDialog)
{
	setAttribute(Qt::WA_DeleteOnClose);
	ui->setupUi(this);
	setFont(QFont(FONTNAMETOUSE));

	ui->tFilesView->setModel(&filesModel);
	ui->tFilesView->setColumnWidth(0, 300);
	ui->tFilesView->setColumnWidth(1, 100);
	ui->tFilesView->setColumnWidth(2, 158);

	selectedStationName = "";
	refreshStations();
	refreshFilesList();
	initTable();
	filesModel.setHorizontalHeaderLabels( QStringList() << tr("Nom") << tr("Taille") << tr("Date de modification"));
	ui->tFilesView->horizontalHeader()->setSectionResizeMode(0, QHeaderView::Stretch);

	connect( ui->tFilesView->horizontalHeader(), SIGNAL(sectionClicked(int)), ui->tFilesView, SLOT(sortByColumn(int)));

	ui->bDownload->setEnabled(false);

	QSettings qsettings( "AMEA", "JOTO" );
	move(qsettings.value("downloadWindowpos", QPoint(200, 200)).toPoint());
	resize(qsettings.value("downloadWindowsize", QSize(600, 450)).toSize());

	connect( ui->tFilesView, &QTableView::doubleClicked, this, &DownloadFileDialog::doubleClicked);
	connect( &notificationCenter, &NotificationCenter::onStationAdded, this, &DownloadFileDialog::refreshStations);
	connect( &notificationCenter, &NotificationCenter::onStationEdited, this, &DownloadFileDialog::refreshStations);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &DownloadFileDialog::refreshStations);
	connect( &notificationCenter, &NotificationCenter::onStationVisibilityChanged, this, &DownloadFileDialog::refreshStations);
}

DownloadFileDialog::~DownloadFileDialog()
{
	QSettings qsettings( "AMEA", "JOTO" );
	qsettings.setValue("downloadWindowsize", size());
	qsettings.setValue("downloadWindowpos", pos());
	delete ui;
	ui = NULL;
	downloadFileDialog = NULL;
}

void DownloadFileDialog::setStationName(const QString &stationName)
{
	if( stationName.isEmpty() && ui->cbStationList->count() > 0)
		selectedStationName = ui->cbStationList->itemText(0);
	else
		selectedStationName = stationName;
	Station *s = getStation();
	if( s)
	{
		ui->cbStationList->setCurrentText(s->getDisplayName());
		refreshFilesList();
		return;
	}
	filesModel.removeRows(0, filesModel.rowCount());
	selectedStationName = "";
	ui->lCurDir->clear();
}

void DownloadFileDialog::refreshStations()
{
	ui->cbStationList->clear();
	ui->cbStationList->setEnabled(true);
	foreach( Station *s, Station::_machines)
		if( s->_isNotConnected == false)
			ui->cbStationList->addItem(s->getDisplayName());
	if( ui->cbStationList->count() == 0)
	{
		filesModel.removeRows(0, filesModel.rowCount());
		ui->cbStationList->setEnabled(false);
		ui->lInfos->setText(tr( "Pas de station visible sur le réseau..."));
		ui->lCurDir->clear();
		ui->bUp->setEnabled(false);
		ui->bHome->setEnabled(false);
		ui->bDisks->setEnabled(false);
	}
	else
	{
		Station *selectedStation = getStation();
		if( selectedStation == NULL)
		{
			filesModel.removeRows(0, filesModel.rowCount());
			ui->lCurDir->clear();
			if( !selectedStationName.isEmpty()) {
				ui->lInfos->setText(tr( "La machine souhaitée n'est plus disponible..."));
			}
			selectedStationName = "";
			ui->bUp->setEnabled(false);
			ui->bHome->setEnabled(false);
			ui->bDisks->setEnabled(false);
		}
		else
		{
			ui->cbStationList->setCurrentText(selectedStation->getDisplayName());
			ui->bUp->setEnabled(true);
			ui->bHome->setEnabled(true);
			ui->bDisks->setEnabled(true);
		}
	}
}

void DownloadFileDialog::itemClicked(void)
{
	ui->bDownload->setEnabled(false);
	for( int i = 0; i < filesModel.rowCount(); i++)
	{
		QStandardItem *item = filesModel.item(i, 0);
		if( item && item->checkState() == Qt::Checked)
		{
			ui->bDownload->setEnabled(true);
			break;
		}
	}
}

void DownloadFileDialog::doubleClicked(const QModelIndex &modelIndex)
{
	Station *s = getStation();
	if( s == NULL)
		return;
	QStandardItem *item = filesModel.item( modelIndex.row(), 0);
	if( item && item->data().toBool() == true)
	{
		if( !s->_currentDirPath.isEmpty() && s->_currentDirPath.right( 1) != "/")
			s->_currentDirPath.append( "/");
		s->_currentDirPath.append( item->text());
		refreshFilesList();
	}
	ui->bDownload->setEnabled(false);
	for( int i = 0; i < filesModel.rowCount(); i++)
	{
		QStandardItem *item = filesModel.item(i, 0);
		if( item && item->checkState() == Qt::Checked)
		{
			ui->bDownload->setEnabled(true);
			break;
		}
	}
}

void DownloadFileDialog::on_cbStationList_activated(const QString &selectedItem)
{
	selectedStationName = selectedItem;
	Station *station = getStation();
	if( station)
	{
		refreshFilesList();
		return;
	}
	selectedStationName = "";
	ui->lCurDir->clear();
	return;
}

void DownloadFileDialog::initTable()
{
	filesModel.removeRows(0, filesModel.rowCount());
}

Station *DownloadFileDialog::getStation()
{
	foreach( Station *station, Station::_machines) {
		if( station->_name == selectedStationName
			|| (!station->_usageName.isEmpty() && station->_usageName == selectedStationName)) {
			return station;
		}
	}
	return NULL;
}

void DownloadFileDialog::refreshFilesList()
{
	initTable();
	Station *selectedStation = getStation();
	if( selectedStation == NULL)
	{
		selectedStationName = "";
		ui->lCurDir->clear();
		ui->bUp->setEnabled(false);
		ui->bHome->setEnabled(false);
		ui->bDisks->setEnabled(false);
		return;
	}
	filesListDownloader.GetFilesList(ui, selectedStation);
}

void DownloadFileDialog::on_bUp_clicked()
{
	Station *selectedStation = getStation();
	if( selectedStation == NULL)
		return;
	QString currentPath = selectedStation->_currentDirPath;
	int index = currentPath.lastIndexOf('/');
	if( index >= 1)
	{
		if( index == 2 && currentPath.at(1) == ':')
			index++;
		QString f = currentPath.left(index);
		if( selectedStation->_currentDirPath != f)
		{
			selectedStation->_currentDirPath = f;
			refreshFilesList();
		}
	}
	else if( index == 0)
	{
		selectedStation->_currentDirPath = "/";
		refreshFilesList();
	}
}

void DownloadFileDialog::on_bDisks_clicked()
{
	initTable();
	ui->lCurDir->clear();
	Station *selectedStation = getStation();
	if( selectedStation == NULL)
	{
		selectedStationName = "";
		return;
	}
	volumesListDownloader.GetVolumesList(ui, selectedStation);
}

void DownloadFileDialog::on_bDownload_clicked()
{
	Station *selectedStation = getStation();
	if( selectedStation == NULL)
	{
		selectedStationName = "";
		ui->lCurDir->clear();
		return;
	}

	QString path = selectedStation->_currentDirPath;
	if( !path.isEmpty() && path.right( 1) != "/")
		path.append( "/");
	QStringList selectedFiles;
	for( int i = 0; i < filesModel.rowCount(); i++)
	{
		QStandardItem *item = filesModel.item(i, 0);
		if( item && item->checkState() == Qt::Checked)
			selectedFiles.append( path + item->text());
	}

	if( selectedFiles.isEmpty())
		return;

	if( LicenseManager::checkForLicenseLevel(1))
	{
		UserAction *newAction = new DownloadFilesAction(selectedStation->_name, selectedStation->getDisplayName(), selectedFiles);
		historique.append( newAction);
		newAction->performAction();
	}
	// pas d'erreur : déselectionner
	for( int i = 0; i < filesModel.rowCount(); i++)
	{
		QStandardItem *item = filesModel.item(i, 0);
		if( item && item->checkState() == Qt::Checked)
			item->setCheckState(Qt::Unchecked);
	}
}

void DownloadFileDialog::on_bClose_clicked()
{
	done(Accepted);
}

void DownloadFileDialog::on_bHome_clicked()
{
	Station *selectedStation = getStation();
	if( selectedStation == NULL)
		return;
	selectedStation->_currentDirPath = "";
	ui->lCurDir->setText( "");
	refreshFilesList();
}

