// ===================================================
// Fichier logdialog.cpp
// Application JOTO
// Copyright (c) AMEA, 2015
// Auteur : AMEA / voir joto-transfert.fr/contact.html
// Diffusé sous license LGPL
// ===================================================

#include "logdialog.h"
#include "ui_logdialog.h"

#include <QCloseEvent>
#include <QStyledItemDelegate>
#include <QPainter>

#include "global.h"

// colonnes :
// Nom, progression, pause, arret
enum Colonnes {
	ColName,
	ColProgress,
	ColPause,
	ColStop
};

QPixmap *LogDialog::stop = NULL;
QPixmap *LogDialog::pause = NULL;

class ProgressBarDelegate : public QStyledItemDelegate
{
public :
	ProgressBarDelegate(TasksTableModel *model) : QStyledItemDelegate(), _model( model) {}

	void paint( QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index ) const
	{
		QStyleOptionProgressBar progressBarStyleOption;
		progressBarStyleOption.rect = option.rect;

		const int progress = _model->getData(index);

		progressBarStyleOption.minimum = 0;
		progressBarStyleOption.maximum = 100;
		progressBarStyleOption.textAlignment = Qt::AlignCenter;
		progressBarStyleOption.progress = progress ;
		progressBarStyleOption.text = QString( "%1 %" ).arg( progress );
		progressBarStyleOption.textVisible = true;

		qApp->style()->drawControl( QStyle::CE_ProgressBar, &progressBarStyleOption, painter );
	}

private :
	TasksTableModel *_model;
};


class ButtonDelegate : public QStyledItemDelegate
{
public :
	ButtonDelegate(QPixmap *pixmap) : QStyledItemDelegate(), _pixmap( pixmap) {}

	void paint( QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& /*index*/ ) const
	{
		painter->drawPixmap(
			QRect(option.rect.left() + option.rect.width() / 2 - 10, option.rect.top() + option.rect.height() / 2 - 10, 20, 20),
			*_pixmap);
	}

private :
	QPixmap *_pixmap;
};



void TasksTableModel::initWithDatas()
{
	// TRES IMPORTANT, car emit dataChanged() va se faire entre threads
	qRegisterMetaType< QVector<int> >("QVector<int>");

	_listTasks.clear();
	foreach( Station *s, Station::_machines)
	{
		foreach( TransfertFile *task, s->_receivingTasks)
			_listTasks.append(QPair<TransfertFileBase *,int>(task, false));

		foreach( TransfertFileBase *task, s->_sendingTasks)
			_listTasks.append(QPair<TransfertFileBase *,int>(task, true));
	}
}

void TasksTableModel::addTask(TransfertFileBase *task, int direction)
{
	beginInsertRows(QModelIndex(), _listTasks.size(), _listTasks.size());
	_listTasks.append( QPair<TransfertFileBase *,int>(task, direction));
	endInsertRows();
	return;
}

void TasksTableModel::removeTask(TransfertFileBase *task)
{
	for( int row = 0; row < _listTasks.size(); row++)
	{
		if( _listTasks[row].first == task)
		{
			beginRemoveRows(QModelIndex(), row, row);
			_listTasks.removeAt(row);
			endRemoveRows();
			return;
		}
	}
	qDebug() << "removeTask : Task not found";
}

void TasksTableModel::updateTask(TransfertFileBase *task)
{
	for( int row = 0; row < _listTasks.size(); row++)
	{
		if( _listTasks[row].first == task)
		{
			emit dataChanged(createIndex(row, ColProgress), createIndex(row, ColProgress));
			return;
		}
	}
	qDebug() << "updateTask : Task not found";
}

int TasksTableModel::getData(const QModelIndex &index) const
{
	if( index.row() < _listTasks.count())
		return _listTasks[index.row()].first->getPourcent();
	return 0;
}

Qt::ItemFlags TasksTableModel::flags(const QModelIndex &index) const
{
	Qt::ItemFlags flags = QAbstractTableModel::flags(index);
	if( index.isValid())
		flags |= Qt::ItemIsEnabled;
	return flags;
}

QVariant TasksTableModel::data(const QModelIndex &index, int role) const
{
	int col = index.column();
	int row = index.row();
	if( !index.isValid()
			|| row < 0 || row >= _listTasks.count()
			|| col < ColName || col > ColProgress)
		return QVariant();

	if( role == Qt::TextAlignmentRole)
	{
		if( col == ColName) return Qt::AlignLeft + Qt::AlignVCenter;
		else return (Qt::AlignHCenter + Qt::AlignVCenter);
	}

	if( role == Qt::DisplayRole || role == Qt::EditRole)
	{
		if( col == ColName) {
			TransfertFileBase *task = _listTasks[row].first;
			QString label = _listTasks[row].second ? "Envoi de " : "Réception de ";
			if( task->_filesList.size() == 1) {
				QFileInfo f(task->_filesList.first());
				label.append( f.fileName());
			}
			else
				label.append( tr("%1 fichiers").arg( task->_filesList.size()));
			if( task->_pauseNeeded || task->_pauseAsked)
				label.append(" (en pause)");
			return label;
		}
	}
	return QVariant();
}

int TasksTableModel::rowCount(const QModelIndex &parent) const
{
	return parent.isValid() ? 0 : _listTasks.count();
}

int TasksTableModel::columnCount(const QModelIndex &parent) const
{
	return parent.isValid() ? 0 : 4;
}

void TasksTableModel::pauseTask(int rowOfTask)
{
	if( rowOfTask < _listTasks.count())
		_listTasks[rowOfTask].first->pause();
	emit dataChanged(createIndex(rowOfTask, ColName), createIndex(rowOfTask, ColName));
}

void TasksTableModel::pauseAskedTask(TransfertFileBase *task)
{
	for( int row = 0; row < _listTasks.size(); row++)
	{
		if( _listTasks[row].first == task)
		{
			emit dataChanged(createIndex(row, ColName), createIndex(row, ColName));
			return;
		}
	}
	qDebug() << "pauseAskedTask : Task not found";
}

void TasksTableModel::stopTask(int rowOfTask)
{
	if( rowOfTask < _listTasks.count())
		_listTasks[rowOfTask].first->stop();
}



LogDialog::LogDialog(QWidget *parent) :
	QDialog(parent),
	ui(new Ui::LogDialog)
{
	if( pause == NULL)
	{
		pause = new QPixmap("://icones/pause_icon.png");
		stop = new QPixmap("://icones/stop_icon.png");
	}

	ui->setupUi(this);
	ui->jobsTableView->setModel(&model);
	ui->jobsTableView->horizontalHeader()->setSectionResizeMode(ColName, QHeaderView::Stretch );
	ui->jobsTableView->setItemDelegateForColumn(ColProgress, new ProgressBarDelegate(&model));
	ui->jobsTableView->setItemDelegateForColumn(ColPause, new ButtonDelegate(pause));
	ui->jobsTableView->setItemDelegateForColumn(ColStop, new ButtonDelegate(stop));
}

LogDialog::~LogDialog()
{
	delete pause; pause = NULL;
	delete stop;  stop = NULL;
	delete ui;
}

void LogDialog::addMessage(const QString &message)
{
	ui->tMessages->addItem( message);
	ui->tMessages->scrollToBottom();
}

void LogDialog::closeEvent(QCloseEvent *e)
{
	setVisible(false);
	e->ignore();
}

void LogDialog::on_jobsTableView_clicked(const QModelIndex &index)
{
	int row = index.row();
	if( !index.isValid() || row < 0) return;

	if( index.column() == ColPause)
		model.pauseTask( row);
	else if( index.column() == ColStop)
		model.stopTask( row);
}

void LogDialog::on_bClose_clicked()
{
	setVisible(false);
}

void LogDialog::on_bClear_clicked()
{
	ui->tMessages->clear();
}
