// ===================================================
// Fichier server.cpp
// Application JOTO
// Copyright (c) AMEA, 2015
// Auteur : AMEA / voir joto-transfert.fr/contact.html
// Diffusé sous license LGPL
// ===================================================

#include "global.h"
#include "server.h"
#include "mainwindow.h"
#include "preferencesdialog.h"
#include "logdialog.h"

#include <QApplication>
#include <QProcess>
#include <QSettings>
#include <QDirIterator>
#include <QClipboard>
#include <QHostInfo>

Clipboard clipboardUpdater;

void Clipboard::setClipboard(const QString &text)
{
	QApplication::clipboard()->setText( text);
}

void CheckStartup( bool startupEnabled)
{
#if defined(Q_OS_WIN)
	QSettings settings("HKEY_CURRENT_USER\\Software\\Microsoft\\Windows\\CurrentVersion\\Run",QSettings::NativeFormat);
	if (startupEnabled) {
		settings.setValue("JOTO", QCoreApplication::applicationFilePath().replace('/','\\'));
	} else {
		settings.remove("JOTO");
	}
#else
	QStringList scriptArgs;
	scriptArgs << QLatin1String("-e");
	if( startupEnabled) {
		scriptArgs << QString("tell application \"System Events\" to make login item at end with properties {path:\"%1\", hidden:false}")
					  .arg( QCoreApplication::applicationDirPath().left(QCoreApplication::applicationDirPath().length() - QStringLiteral("Contents/MacOS").length()));
	} else {
		scriptArgs << QString("tell application \"System Events\" to delete login item \"JOTO\"");
	}
	QProcess::execute(QLatin1String("/usr/bin/osascript"), scriptArgs);
#endif
}

void openDir(QFileInfo dirToShow)
{
	// Mac, Windows support folder or file.
#if defined(Q_OS_WIN)
	QString param;
	if (!dirToShow.isDir())
		param = QLatin1String("/select,");
	param += QDir::toNativeSeparators(dirToShow.filePath());
	QProcess::startDetached("explorer.exe", QStringList(param));
#elif defined(Q_OS_MAC)
	QStringList scriptArgs;
	scriptArgs << QLatin1String("-e")
			   << QString::fromLatin1("tell application \"Finder\" to reveal POSIX file \"%1\"")
									 .arg(dirToShow.absoluteFilePath());
	QProcess::execute(QLatin1String("/usr/bin/osascript"), scriptArgs);
	scriptArgs.clear();
	scriptArgs << QLatin1String("-e")
			   << QLatin1String("tell application \"Finder\" to activate");
	QProcess::execute("/usr/bin/osascript", scriptArgs);
#else
	// not supported
#endif

}

void openApplicationDir()
{
	static QFileInfo fileToShow( QCoreApplication::applicationDirPath() + "/" + APPSUFFIX2);
	openDir(fileToShow);
}

void openReceiveBox()
{
	static QFileInfo fileToShow( QDir::homePath().append("/reception"));
	openDir(fileToShow);
}

QString convertFromUFN( QString &fileInUFNNotation)
{
	if( fileInUFNNotation.mid(0, strlen(STATIONOS)) == QStringLiteral(STATIONOS))
		return fileInUFNNotation.mid( strlen(STATIONOS));
	QChar *p = fileInUFNNotation.data();
	while( *p != ':' && !p->isNull()) p++;
	if( !p->isNull())
	{
		p++;
		QChar *start = p;
		while( !p->isNull())
		{
#ifdef Q_OS_WIN
			if( *p == '/') *p = '\\';
#else
			if( *p == '\\') *p = '/';
#endif
			p++;
		}
		return QString( start);
	}
	return fileInUFNNotation;
}


Server::Server( ) : QThread(), _abordRequested(false)
{
	start();
}

void Server::startServer()
{
	qDebug() << "Server started";
	_server.bindall(PORTNUMBER);
	while( !_abordRequested) {
		ipstream *client = new ipstream();
		if( _server.serve(*client, -1, 1000))
		{
			// When a new client connects, the server constructs a QTcpSocket and all
			// communication with the client is done over this QTcpSocket.
			//qDebug() << "incoming connection";
			ConnectionThread *conn = new ConnectionThread(this, client);
			_connections.append(conn);
		}
		else
			delete client;
	}
}

void Server::onConnexionFinished(ConnectionThread *conn)
{
	_connections.removeOne(conn);
}

void Server::stop()
{
	foreach(ConnectionThread *c, _connections)
		c->stop();
	_connections.clear();
	_abordRequested = true;
	qDebug() << "Server stopped";
}

void ConnectionThread::onEndThread() {
	if( _server)
		_server->onConnexionFinished(this);
	deleteLater();
}


void ConnectionThread::stop()
{
	_exitLoop = true;
	while( _exitPerformed == false)
		msleep(100);
}

void ConnectionThread::stationClosedEvent(Station *stationClosed)
{
	qDebug() << "ConnectionThread::stationClosedEvent";
	if( stationClosed == _station)
		_station = NULL;
}

void ConnectionThread::run()
{
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &ConnectionThread::stationClosedEvent, Qt::DirectConnection);
	connect(this, &ConnectionThread::setClipboardInMainThread, &clipboardUpdater, &Clipboard::setClipboard);

	qDebug() << "ConnectionThread RUN";
	_station = NULL;
	_connection = new ExtendedSocket(_socket);
	if( _connection->waitForConnectedWithTimer() == false)
	{
		delete _connection;
		qDebug() << "Pas de connexion possible, arret ConnectionThread.";
		return;
	}
	QHostAddress address = _connection->peerAddress();
	qDebug() << address;

	try
	{
		_ident = _connection->readLine().toUpper();
		if( _ident.isEmpty())
			_exitLoop = true;
		else
		{
			QString usageName = _connection->readLine();
			int iconeIndex = _connection->readLine().toInt();
			QString stationOS = _connection->readLine();
			int stationVersion = _connection->readLine().toInt();
			qDebug() << "contacted by" << address.toString() << "(" << _ident << "," << usageName << "," << iconeIndex << "," << stationOS << "," << stationVersion << ")";
			{
				qDebug() << "lock1 addStation";
				QMutexLocker locker( &Station::stationMutex);
				_station = Station::findStation( _ident, usageName);
				if( _station == NULL && _ident != Station::_localStation->_name)
				{
					qDebug() << "New machine.";
					_station = new Station( _ident, address, usageName, iconeIndex, stationOS, stationVersion);
					emit notificationCenter.onStationAdded( _station);
				}
				qDebug() << "unlock1";
			}
			if( _station)
			{
				if( _station->_isNotConnected == true)
				{
					//statutMessages += tr("Retour de %1").arg(_station->getDisplayName());
					_station->_isNotConnected = false;
					emit notificationCenter.onStationVisibilityChanged(_station);
				}
				if( _station->_usageName != usageName || _station->_iconeIndex != iconeIndex)
				{
					_station->_usageName = usageName;
					_station->_iconeIndex = iconeIndex;
					if( mainWindow)
						mainWindow->updateMachines();
				}
				if( updateChecked == false && _station->_stationVersion > VERSION) {
					updateChecked = true;
					if( mainWindow)
						emit mainWindow->startAutoUpdateSignal("Des versions différentes de JOTO sont présentes sur votre réseau. Une mise à jour est vivement conseillée.",true, true);
				}
			}
		}
		while(_exitLoop == false)
		{
			QString cmd = _connection->readLine();
			qDebug() << "cmd =" << cmd;

			if( cmd.contains( "setclipboard", Qt::CaseInsensitive))
			{
				QString valueClipboard = _connection->readLine();
				emit setClipboardInMainThread(valueClipboard);
				statutMessages += tr("Contenu du resse-papier bien reçu.");
/*				if( useSounds) {
					MyPlaySound(":/sons/son2.wav");
				}*/
			}
			else if(cmd.contains("getclipboard", Qt::CaseInsensitive))
			{
				QString clipText = QApplication::clipboard()->text();
				_connection->sendLine(clipText);
				statutMessages += tr("Envoi du contenu du presse-papier");
			}
			else if( cmd.contains( "openreceivebox", Qt::CaseInsensitive))
			{
				if( autoOpen)
					openReceiveBox();
				/*if( useSounds) {
					MyPlaySound(":/sons/son1.wav");
				}*/
			}
			else if( cmd.contains("sendmeupdate"))
			{
				QFileInfo file(QCoreApplication::applicationDirPath() + APPSUFFIX);
				QString fName = file.canonicalFilePath();
				_station->addTransfertTask(QStringList() << fName);
			}
			else if( cmd.contains( "sendmefiles"))
			{
				bool ok1 = false;
				QString nbFilesTxt = _connection->readLine();
				int nbFiles = nbFilesTxt.toInt(&ok1);
				if( ok1)
				{
					QStringList filesList;
					for( int i = 0; i < nbFiles; i++)
						filesList << _connection->readLine();
					_station->addTransfertTask( filesList);
				}
			}
			else if( cmd.contains( "cleanup", Qt::CaseInsensitive))
			{
				QString fileName = _connection->readLine();
				QString fileNameString = convertFromUFN( fileName);
				QDir home( QDir::homePath());
				QFileInfo file( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( fileNameString));
				QDir d(file.absoluteFilePath());
				statutMessages += "cleanup " + file.absoluteFilePath();
				d.removeRecursively();
			}
			else if( cmd.contains( "dropfile", Qt::CaseInsensitive))
			{
				bool ok1 = false, ok2 = false;
				QString fileName = _connection->readLine();
				QString nbMsgsTxt = _connection->readLine();
				int nbMsgs = nbMsgsTxt.toInt(&ok1);
				QString lastMsgSizeTxt = _connection->readLine();
				int lastMsgSize = lastMsgSizeTxt.toInt(&ok2);
				QString permissionsTxt = _connection->readLine();

				if( ok1 && ok2)
				{
					QString fileNameString = convertFromUFN( fileName);
					// le placer dans un répertoire spécial nommé Réception
					QDir home( QDir::homePath());
					if( home.exists("reception") == false)
						home.mkdir("reception");

					QFileInfo file( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( fileNameString));
					QDir parentDir( file.path());
					if( parentDir.exists() == false)
						parentDir.mkpath(".");
					qDebug() << "receiving a file :" << file.absoluteFilePath();

					if( file.exists())
					{
						QString newFileName = QString("%1/%2 %3")
								.arg(file.path(), file.baseName(), file.lastModified().toString("dd_MM_yyyy hh_mm_ss"));
						if( !file.completeSuffix().isEmpty())
							newFileName.append(".").append(file.completeSuffix());
						QFile f( file.absoluteFilePath());
						f.rename( newFileName);
					}

					FILE *f = fopen( file.absoluteFilePath().toLocal8Bit(), "wb");
					if( f)
					{
						_connection->sendLine( "READY");
						bool writingOK = true;
						QString errorMsg = "";
						long long fileSize = nbMsgs * TRANSFERT_BUFF_SIZE + lastMsgSize;
						TransfertFileBase *newTask = new TransfertFileBase(QStringList() << file.absoluteFilePath(), fileSize);
						if( _station)
						{
							_station->_sendingTasks.append( newTask);
							if( logDialog)
								logDialog->addTask( newTask, 0);
							if( mainWindow)
								mainWindow->updateMachines();
						}

						try {
							for( int i = 0; i <= nbMsgs && writingOK; i++)
							{
								unsigned int sizeToDownload = TRANSFERT_BUFF_SIZE;
								if( i == nbMsgs)
									sizeToDownload = lastMsgSize;
								//msleep(10);
								if( _connection->state() != QAbstractSocket::ConnectedState)
								{
									errorMsg = tr("Perte du réseau durant l'attente de données");
									qDebug() << errorMsg;
									writingOK = false;
								}
								if( _exitLoop)
								{
									errorMsg = tr("Arret demandé");
									qDebug() << errorMsg;
									writingOK = false;
								}
								if( writingOK && sizeToDownload > 0)
								{
									QString msg = _connection->readLine();
									if( msg == "Pause")
									{
										newTask->_pauseAsked = true;
										if( logDialog)
											logDialog->pauseAskedTask(newTask);
										msleep(1000);
										i--;
										_connection->sendLine("pause OK");
										continue;
									}
									else if( msg != "Sending") {
										errorMsg = msg;
										qDebug() << errorMsg;
										writingOK = false;
									} else {
										if( newTask->_pauseAsked) {
											newTask->_pauseAsked = false;
											if( logDialog)
												logDialog->pauseAskedTask(newTask);
										}
										QByteArray bytes = _connection->readBuffer();
										unsigned int size = bytes.size();
										newTask->_currentSize += size;
										if( logDialog)
											logDialog->updateTask( newTask);
										if( fwrite( bytes.constData(), 1, size, f) != size) {
											errorMsg = tr("Erreur d'écriture des données reçues");
											qDebug() << errorMsg;
											writingOK = false;
										}
									}
								}
								if( writingOK == false) {
									if( _connection->state() == QAbstractSocket::ConnectedState)
										_connection->sendLine(errorMsg);
								}
								else if( sizeToDownload > 0)
								{
									if( newTask->_exitLoopRequested)
									{
										errorMsg = tr("Transfert interrompu");
										qDebug() << errorMsg;
										_connection->sendLine( errorMsg);
										writingOK = false;
										_exceptionToSkip = true;
									}
									else if( newTask->_pauseNeeded)
									{
										_connection->sendLine("pause");
										msleep(1000);
										QString ack = _connection->readLine();
										if( ack != "pause OK")
										{
											qDebug() << "ERREUR DURANT PAUSE" << ack;
											errorMsg = tr("Erreur durant la pause");
											statutMessages += tr( "Transfert de %1 interrompu : erreur durant la pause : %2").arg(file.fileName()).arg(ack);
											writingOK = false;
										}
									}
									else
										_connection->sendLine( i == nbMsgs ? "endOK" : "partOK");
								}
							}
							fclose( f);
							f = NULL;
							if( _station) {
								if( logDialog)
									logDialog->removeTask(newTask);
								_station->_sendingTasks.removeOne(newTask);
								if( mainWindow)
									mainWindow->updateMachines();
							}
							delete newTask;
						}
						catch(QString msg)	// en cas d'erreur supprimer le fichier en cours de transfert
						{
							qDebug() << "Erreur catchée, supprimer le fichier" << msg;
							if( f)
							{
								fclose( f);
								f = NULL;
							}
							QDir d(file.path());
							d.remove( file.fileName());
							if( _station)
							{
								if( logDialog)
									logDialog->removeTask(newTask);
								_station->_sendingTasks.removeOne(newTask);
								if( mainWindow)
									mainWindow->updateMachines();
							}
							delete newTask;
							throw msg;
						}

						if( writingOK)
							statutMessages += tr("Réception de %1 OK.").arg(file.fileName());
						else
							statutMessages += tr("Echec de la réception de %1 : %2.").arg(file.fileName()).arg(errorMsg);
						if( writingOK == false)
						{
							QDir d(file.path());
							d.remove( file.fileName());
						}
#ifdef Q_OS_MAC
						if( writingOK && fileName.left( strlen(STATIONOS)) == QStringLiteral(STATIONOS)) // ne le faire que sur un transfert entre Macs
						{
							QFile f(file.absoluteFilePath());
							qDebug() << "permissions : " << QFile::Permissions( permissionsTxt.toInt());
							f.setPermissions( QFile::Permissions( permissionsTxt.toInt()));
						}
#endif
						if( writingOK == true) {
							nbFilesReceived++;
						}
					}
					else
					{
						_connection->sendLine( tr("Erreur : impossible d'ouvrir le nouveau fichier en écriture"));
						statutMessages += tr("Réception de %1 : impossible d'ouvrir le fichier").arg(file.fileName());
					}
				}
				else
					qDebug() << "dropfile : arguments incorrects : " << fileName << ", " << nbMsgsTxt << ", " << lastMsgSizeTxt << ", " << permissionsTxt;
			}
			else if( cmd.contains( "makedir", Qt::CaseInsensitive))
			{
				QString dirName = _connection->readLine();

				_connection->sendLine( "OK");

				QString fileNameString = convertFromUFN( dirName);
				// le placer dans un répertoire spécial nommé Réception
				QDir home( QDir::homePath());
				if( home.exists("reception") == false)
					home.mkdir("reception");

				QFileInfo dir( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( fileNameString));
				qDebug() << "creating dir  :" << dir.absolutePath() << dir.fileName();

				if( dir.exists())
				{
					QString newDirName = QString("%1/%2 %3.%4")
							.arg(dir.path(), dir.baseName(), dir.lastModified().toString("dd MM yyyy hh mm ss"), dir.completeSuffix());
					QFile d( dir.absoluteFilePath());
					d.rename( newDirName);
				}

				dir.absoluteDir().mkpath(dir.fileName());
			}
			else if( cmd.contains( "makelink", Qt::CaseInsensitive))
			{
				QString linkName = _connection->readLine();
				QString linkTarget = _connection->readLine();

				_connection->sendLine( "OK");

				QString linkNameString = convertFromUFN( linkName);
				QString targetNameString = convertFromUFN( linkTarget);
				// le placer dans un répertoire spécial nommé Réception
				QDir home( QDir::homePath());
				if( home.exists("reception") == false)
					home.mkdir("reception");

				QFile linkFile( QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( linkNameString));
				QString targetTxt = QString("%1/%2").arg( home.absoluteFilePath("reception")).arg( targetNameString);
				qDebug() << "link : " << linkFile.fileName() << " -> " << targetTxt;
				linkFile.link( targetTxt);
			}
			else if( cmd.contains( "newports", Qt::CaseInsensitive))
			{
				QString tcpPortTxt = _connection->readLine();
				QString udpPortTxt = _connection->readLine();
				bool ok1 = false, ok2 = false;
				int value1 = tcpPortTxt.toInt( &ok1);
				int value2 = udpPortTxt.toInt( &ok2);
				if( ok1 && ok2)
					emit mainWindow->changePortsSignal(value1, value2);
				else
					qDebug() << "newport : arguments incorrects : " << tcpPortTxt << ", " << udpPortTxt;
			}
			else if( cmd.contains( "listfiles", Qt::CaseInsensitive))
			{
				QString dirToRead = _connection->readLine();
				if( dirToRead.isEmpty())
					dirToRead = homeDir;
				_connection->sendLine(dirToRead);
				qDebug() << "reading " << dirToRead;
				if( canProcessDir( dirToRead))
				{
					QDirIterator dirIt(dirToRead);
					while( dirIt.hasNext())
					{
						dirIt.next();
						_connection->sendLine( QStringLiteral("file"));
						QFileInfo fInfo = dirIt.fileInfo();
						_connection->sendLine( fInfo.fileName());
						_connection->sendLine( QString::number( fInfo.size()));
						_connection->sendLine( fInfo.lastModified().toString("dd/MM/yyyy hh:mm:ss"));
						_connection->sendLine( (fInfo.isDir() ? "1" : "0"));
					}
					_connection->sendLine( QStringLiteral("end"));
				}
				else
				{
					_connection->sendLine( QStringLiteral("message"));
					_connection->sendLine( tr("L'acces a ce repertoire n'est pas autorisé"));
				}
			}
			else if( cmd.contains("listvolumes", Qt::CaseInsensitive))
			{
				foreach(const QFileInfo &driveInfo, QDir::drives())
					_connection->sendLine( driveInfo.absoluteFilePath());

				_connection->sendLine( QStringLiteral("end"));
			}
			else if( cmd.contains( "updatelicense", Qt::CaseInsensitive))
			{
				QString key = _connection->readLine();
				QString value = _connection->readLine();
				int prevLicenceLevel = LicenseManager::getLicenseLevel();
				LicenseManager::setKeyAndValue(key, value);
				if( LicenseManager::getLicenseLevel() > prevLicenceLevel)
					emit mainWindow->closeLicenceMessageDialog();
			}
			else if( cmd.contains( "asklicense", Qt::CaseInsensitive))
			{
				_connection->sendLine(LicenseManager::getKey());
				_connection->sendLine(LicenseManager::getValue());
				_connection->sendLine(LicenseManager::getDate().toString());
				_connection->sendLine(LicenseManager::getFirstRunDate().toString());
			}
			else if( cmd.contains( "upgradeauto", Qt::CaseInsensitive))
			{
				if( mainWindow)
					emit mainWindow->startAutoUpdateSignal("",true, true);
			}
			else if( cmd.contains( "close", Qt::CaseInsensitive))
			{
				_exitLoop = true;
			}
			else if( cmd.contains( "quit", Qt::CaseInsensitive))
			{
				qDebug() << "quit";
				Station::removeStation( _ident);
			}
			else {
				qDebug() << "Commande ignoree.";
				if( _connection->error())
				{
					qDebug() << _connection->errorString();
					_exitLoop = true;
				}
			}
		}
		_connection->disconnectFromHost();
		delete _connection;
	}
	catch(QString msg)
	{
		qDebug() << "EXCEPTION dans ConnectionThread : " + msg;
		if( _exceptionToSkip == false)
			statutMessages += tr("Erreur réseau : ") + msg;
		_connection->disconnectFromHost();
		delete _connection;
	}
	_exitPerformed = true;
	qDebug() << "ConnectionThread END";
}

bool ConnectionThread::canProcessDir( const QString &dirToRead)
{
	if( accessType == 2)
		return true;
	else if( accessType == 3)
		return false;
	QDir directory( dirToRead);
	directory.makeAbsolute();
	QString directoryString = directory.absolutePath();
	if( !directoryString.endsWith("/"))
		directoryString.append("/");
	foreach( const QString &dir, accessDirList)
	{
		QDir dirToTest( dir);
		dirToTest.makeAbsolute();
		QString dirToTestString = dirToTest.absolutePath();
		if( !dirToTestString.endsWith("/"))
			dirToTestString.append("/");

		bool ok = directoryString.startsWith( dirToTestString);
		if( accessType == 1)
		{
			if( ok == true)
				return false;
		}
		else
		{
			if( ok == true)
				return true;
		}
	}
	if( accessType == 1)
		return true;
	return false;
}



QList<QHostAddress> UDPValidateStation::_pendingValidations;

UDPValidateStation::UDPValidateStation(const QString &stationName, const QString &stationUsageName, int stationIconIndex, QHostAddress senderAddress, QString stationOS, int stationVersion)
	: _stationName( stationName), _stationUsageName(stationUsageName), _stationIconIndex( stationIconIndex), _stationOS(stationOS), _stationVersion(stationVersion), _senderAddress(senderAddress), _socket(NULL)
{
	qDebug() << "Decouverte UDP de la station" << _stationName << "-" << _senderAddress << ", vérification";
	if( _pendingValidations.contains(_senderAddress))
	{
		qDebug() << "vérification annulée, déjà en cours...";
		return;
	}
	_pendingValidations.append(_senderAddress);
	connect( this, &UDPValidateStation::finished, this, &UDPValidateStation::deleteLater);
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &UDPValidateStation::abort, Qt::DirectConnection);
	start();
}

void UDPValidateStation::run()
{
	ExtendedSocket socket;
	Station *s = NULL;
	_socket = &socket;
	try
	{
		{
			qDebug() << "lock1 addStation par UDP";
			QMutexLocker locker( &Station::stationMutex);

			qDebug() << "Vérification OK, station toujours manquante ?";
			Station *station = Station::findStation( _stationName, _stationUsageName);
			if( station == NULL && _stationName != Station::_localStation->_name)
			{
				qDebug() << "Toujours manquante : ajout";
				s = new Station( _stationName, _senderAddress, _stationUsageName, _stationIconIndex, _stationOS, _stationVersion);
				s->_isNotConnected = true;
				s->_connectionLostTimer.start();
				emit notificationCenter.onStationAdded( s);
			}
			else
				qDebug() << "Déjà ajoutée ou LOCALHOST, annulation...";

			qDebug() << "unlock1 addStation par UDP";
		}
		socket.connectToHost(_senderAddress.toString(), PORTNUMBER);
		if( socket.waitForConnectedWithTimer())
		{
			Station::_localStation->sendInfosTo(socket);
			socket.closeConnection();
			if( s) {
				s->_isNotConnected = false;
				emit notificationCenter.onStationVisibilityChanged(s);
			}
		}
		else
		{
			throw( socket.errorString());
		}
	}
	catch( QString msg)
	{
		qDebug() << "Echec connexion vérification : " << msg;
	}
	_pendingValidations.removeOne(_senderAddress);
	_socket = NULL;
}

void UDPValidateStation::abort()
{
	qDebug() << "UDPValidateStation::abort()";
	if( _socket)
		_socket->abort();
}


UDPService::UDPService()
{
	// timer pour l'envoi de messages broadcast de signalement
	_timerUDP.setInterval(1000);
	connect( &_timerUDP, &QTimer::timeout, this, &UDPService::onTimerUDP);
	_timerUDP.start();

	_socketReceiver.bind(QHostAddress::AnyIPv4, BROADCASTPORTNUMBER);
	connect( &_socketReceiver, &QUdpSocket::readyRead, this, &UDPService::readPendingDatagrams);

	connect( &notificationCenter, &NotificationCenter::onStationAdded, this, &UDPService::newStationEvent);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &UDPService::stationClosedEvent, Qt::DirectConnection);
}

void UDPService::onTimerUDP()
{
	QUdpSocket socketSender;
	socketSender.bind(QHostAddress::AnyIPv4, BROADCASTPORTNUMBER, QUdpSocket::ShareAddress);

	QByteArray hostDatagram = (Station::_localStation->_name
							   + "\n" + Station::_localStation->_usageName
							   + "\n" + QString::number((Station::_localStation->_iconeIndex))
							   + "\n" + Station::_localStation->_stationOS
							   + "\n" + QString::number(VERSION) ).toLocal8Bit();
	socketSender.writeDatagram(hostDatagram.data(), hostDatagram.size(), QHostAddress::Broadcast, BROADCASTPORTNUMBER);
}


void UDPService::stopService()
{
	_timerUDP.stop();
	_socketReceiver.close();
}

void UDPService::readPendingDatagrams()
{
	while( _socketReceiver.hasPendingDatagrams())
	{
		QHostAddress sender;
		quint16 senderport;
		QByteArray datagram;
		datagram.resize( _socketReceiver.pendingDatagramSize());

		_socketReceiver.readDatagram( datagram.data(), datagram.size(), &sender, &senderport);
		//qDebug() << "UDP : " << datagram.data();
		QList<QByteArray> datagrams = datagram.split('\n');

		QString peerName = datagrams.first().data();
		peerName = peerName.simplified().toUpper();
		QString peerUsageName = datagrams.at(1).data();
		peerUsageName = peerUsageName.simplified();
		QString stationIconIndexTxt = datagrams.at(2).data();
		int stationIconIndex = stationIconIndexTxt.toInt();
		QString stationOSTxt = datagrams.at(3).data();
		QString stationVersionTxt = datagrams.at(4).data();
		int stationVersion = stationVersionTxt.toInt();
		if( peerName != Station::_localStation->_name)
		{
			Station *station = Station::findStation( peerName, peerUsageName);
			if( station == NULL)
				new UDPValidateStation(peerName, peerUsageName, stationIconIndex, sender, stationOSTxt, stationVersion);
		}
	}
}

void UDPService::newStationEvent(Station *station)
{
	qDebug() << "add " << station->_name << "at" << QTime::currentTime();
	new TestStation(station);
}

void UDPService::stationClosedEvent( Station *station)
{
	qDebug() << "closed " << station->_name << "at" << QTime::currentTime();
}



TestStation::TestStation(Station *s) : _station( s), _socket(NULL)
{
	_timerTest.setSingleShot(true);
	_timerTest.start(5000);
	qDebug() << "Test station " << s->getDisplayName() << s->_address.toString();
	connect( &_timerTest, &QTimer::timeout, this, &TestStation::startTest);
	connect( this, &QThread::finished, this, &TestStation::restartOrStop);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &TestStation::onStationDeleted, Qt::DirectConnection);
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &TestStation::abort);
}

TestStation::~TestStation()
{
	qDebug() << "End TestStation";
}

void TestStation::restartOrStop()
{
	if( _station)
		_timerTest.start(5000);
	else
		deleteLater();
}

void TestStation::startTest()
{
	if( _station)
		start();
	else
		deleteLater();
}

void TestStation::run()
{
	qDebug() << "Start test station " << _station->getDisplayName();

	ExtendedSocket socket;
	_socket = &socket;
	try
	{
		if( _station == NULL) { _socket = NULL; return; }
		socket.connectToHost(_station->_address.toString(), PORTNUMBER);
		if( socket.waitForConnectedWithTimer())
		{
			if( _station == NULL) { _socket = NULL; return; }

			qDebug() << "on connected " << _station->getDisplayName();
			if( _station->_isNotConnected == true)
			{
				statutMessages += tr("Retour de %1").arg(_station->getDisplayName());
				_station->_isNotConnected = false;
				emit notificationCenter.onStationVisibilityChanged(_station);
			}
			Station::sendInfosTo(socket);
			socket.closeConnection();
		}
		else
		{
			throw( socket.errorString());
		}
	}
	catch( QString &msg)
	{
		if( _station == NULL) { _socket = NULL; return; }
		qDebug() << "TestStation::onError " << _station->getDisplayName() << " : " << msg;
		//statutMessages += "TestStation::onError " + _station->getDisplayName();
		if( _station->_isNotConnected == false) {
			statutMessages += tr("Disparition de %1 : %2").arg(_station->getDisplayName()).arg(msg);
			_station->_isNotConnected = true;
			_station->_connectionLostTimer.start();
			emit notificationCenter.onStationVisibilityChanged(_station);
		}
		else {
			if( _station->_connectionLostTimer.elapsed() > TIMEOUT * 2) {
				Station::removeStation(_station->_name);
				socket.abort();
				_socket = NULL;
				return;
			}
		}
		socket.abort();
	}
	_socket = NULL;
}

void TestStation::onStationDeleted(Station *s)
{
	if( s == _station)
		abort();
}

void TestStation::abort()
{
	if( _station)
		qDebug() << "TestStation::abort()" << _station->getDisplayName();
	if( _station && _socket)
		_socket->abort();
	_station = NULL;
}

