// ===================================================
// Fichier tools.cpp
// Application JOTO
// Copyright (c) AMEA, 2015
// Auteur : AMEA / voir joto-transfert.fr/contact.html
// Diffusé sous license LGPL
// ===================================================

#include "global.h"

#include "mainwindow.h"
#include "logdialog.h"

#include <QHostInfo>
#include <QResource>
#include <QApplication>
#include <QMessageBox>
#include <QDesktopServices>
#include <QCryptographicHash>
#include <QElapsedTimer>
#include <QPushButton>

QList<Station *>Station::_machines;
Station *Station::_localStation = NULL;
QMutex Station::stationMutex(QMutex::Recursive);

bool stationLessThan(Station *s1, Station *s2)
{
	return s1->getDisplayName() < s2->getDisplayName();
}

void Station::removeStation(const QString &name)
{
	Station *closingStation = Station::findStation(name);
	if( closingStation)
	{
		Station::_machines.removeOne(closingStation);
		emit notificationCenter.onStationDeleted( closingStation);
		qDebug() << "EFFACEMENT STATION";
		delete closingStation;
	}
}

Station *Station::findStation(const QString &name)
{
	foreach( Station *s, Station::_machines)
	{
		if( name.compare(s->_name, Qt::CaseInsensitive) == 0)
			return s;
	}
	return NULL;
}

Station *Station::findStation(const QString &name, const QString &usageName)
{
	foreach( Station *s, Station::_machines)
	{
		if( name.compare(s->_name, Qt::CaseInsensitive) == 0)
			return s;
		if( !usageName.isEmpty() && !s->_usageName.isEmpty() && usageName.compare(s->_usageName, Qt::CaseInsensitive) == 0)
			return s;
	}
	return NULL;
}

void Station::sendInfosTo(ExtendedSocket &socket)
{
	socket.sendLine( Station::_localStation->_name );
	socket.sendLine( Station::_localStation->_usageName );
	socket.sendLine( QString::number(Station::_localStation->_iconeIndex));
	socket.sendLine( QStringLiteral(STATIONOS));
	socket.sendLine( QString::number(VERSION));
}

Station::Station(const QString &name, const QHostAddress &address, bool isLocal)
	: _name( name.toUpper()), _usageName( ""), _iconeIndex(-1), _stationOS(STATIONOS), _stationVersion(VERSION), _address( address.toIPv4Address()), _isStationAlive( false), _isLocalStation( isLocal), _isNotConnected(false)
{
	if( _isLocalStation)
		_localStation = this;
	else
	{
		qDebug() << "ADD ADDRESS : " << address.toString() << address.protocol();
		if( address.protocol() == QAbstractSocket::IPv4Protocol)
			stationsContactHistory[ address.toString()] = 0;
		_machines.append( this);
		qStableSort( _machines.begin(), _machines.end(), stationLessThan);
	}
}

Station::Station(const QString &name, const QHostAddress &address, QString usageName, int iconeIndex, QString stationOS, int stationVersion)
	: _name( name.toUpper()), _usageName( usageName), _iconeIndex(iconeIndex), _stationOS( stationOS), _stationVersion( stationVersion), _address( address.toIPv4Address()), _isStationAlive( false), _isLocalStation( false), _isNotConnected(false)
{
	qDebug() << "ADD ADDRESS : " << address.toString() << address.protocol();
	if( address.protocol() == QAbstractSocket::IPv4Protocol)
		stationsContactHistory[ address.toString()] = 0;
	_machines.append( this);
	qStableSort( _machines.begin(), _machines.end(), stationLessThan);
}

QPixmap *Station::getPixmap()
{
	static QPixmap *pcPixmap = NULL;
	if( _iconeIndex >= 0)
	{
		if( fileToIcon[filesList.at(_iconeIndex)] == NULL)
			fileToIcon[filesList.at(_iconeIndex)] = new QPixmap(QStringLiteral(FILEPATH) + filesList.at( _iconeIndex));
		return fileToIcon[filesList.at(_iconeIndex)];
	}
	if( pcPixmap == NULL)
		pcPixmap = new QPixmap(pcIcon.pixmap(128,128));
	return pcPixmap;
}

void Station::setUsageNameAndDispath( const QString &usageName, int iconeIndex)
{
	if( _usageName == usageName && _iconeIndex == iconeIndex)
		return;

	_usageName = usageName;
	_iconeIndex = iconeIndex;
	qStableSort( _machines.begin(), _machines.end(), stationLessThan);
	emit notificationCenter.onStationEdited( this);
	sendInfosToAllStations();
}

QString &Station::getDisplayName()
{
	if( _usageName.isEmpty())
		return _name;
	return _usageName;
}

void Station::addTransfertTask(const QStringList &filesNames)
{
	qDebug() << "add transfert files task " << filesNames;
	TransfertFile *newTask = new TransfertFile(filesNames, this);
	if( mainWindow)
		mainWindow->updateMachines();
	newTask->start();
}

bool Station::pendingTasks()
{
	foreach( Station *s, Station::_machines)
	{
		if( s->_receivingTasks.size() + s->_sendingTasks.size() > 0)
			return true;
	}
	return false;
}

void Station::stopTasks()
{
	foreach( Station *s, Station::_machines)
	{
		foreach( TransfertFile *task, s->_receivingTasks)
			task->stop();
		foreach( TransfertFileBase *task, s->_sendingTasks)
			task->stop();

		while( s->_receivingTasks.size() + s->_sendingTasks.size() > 0)
			QThread::msleep(50);
	}
}

void Station::sendInfosToAllStations()
{
	foreach( Station *s, Station::_machines)
	{
		if( s->_isNotConnected)
			continue;
		try
		{
			ExtendedSocket _socket;
			_socket.connectToHost(s->_address.toString(), PORTNUMBER);
			if( _socket.waitForConnectedWithTimer())
			{
				sendInfosTo(_socket);
				_socket.closeConnection();
			}
		}
		catch(QString msg)
		{
			statutMessages += tr( "Erreur réseau durant l'envoi des informations aux autres postes (sendInfosToAllStations) : ") + msg;
		}
	}
}

TransfertFileBase::TransfertFileBase(const QStringList &filesNames, long long fileSize)
	: _filesList( filesNames), _totalSize(fileSize), _currentSize( 0), _pauseNeeded( false), _pauseAsked( false), _exitLoopRequested( false)
{
}

int TransfertFileBase::getPourcent()
{
	long long total = _totalSize;
	long long current = _currentSize;
	if( total > 1024*1024*1024) { total /= 1024 * 1024; current /= 1024 * 1024; }
	int pourcent = 0;
	if( total > 0)
		pourcent = current * 100 / total;
	return pourcent;
}

void TransfertFileBase::pause()
{
	_pauseNeeded = !_pauseNeeded;
}

void TransfertFileBase::stop()
{
	_exitLoopRequested = true;
}

TransfertFile::TransfertFile(const QStringList &filesNames, Station *_destStation)
	: QThread(), TransfertFileBase( filesNames, 0), _station( _destStation), _cleanupNeeded(false),
	  _exitLoopPerformed( false)
{
	foreach( QString f, _filesList)
		_totalSize += dirSize( f);
	connect( &notificationCenter, &NotificationCenter::onStationDeleted, this, &TransfertFile::onStationDeletedNotification, Qt::DirectConnection);
	connect( this, &TransfertFile::finished, this, &TransfertFile::onEndThread, Qt::DirectConnection);
	_station->_receivingTasks.append( this);
}

void TransfertFile::onEndThread()
{
	_station->_receivingTasks.removeOne( this);
	deleteLater();
	if( mainWindow)
		mainWindow->updateMachines();
}

void TransfertFile::onStationDeletedNotification(Station *station)
{
	//qDebug() << "TransfertFile::onStationDeletedNotification";
	if( station == _station)
	{
		stop();
		qDebug() << "onStationDeletedNotification wait";
		while( _exitLoopPerformed == false)
			msleep( 100);
		qDebug() << "onStationDeletedNotification end";
	}
}


bool TransfertFile::doSendDir( const QString &root, QDir &file, ExtendedSocket *stream, Station *station)
{
	try
	{
		stream->sendLine( "makedir");
		stream->sendLine( STATIONOS + file.absolutePath().mid(root.length()));

		QString ack = stream->readLine();
		qDebug() << "makedir => " << ack;
		if( ack != "OK")
		{
			statutMessages += tr( "Création du dossier %1 non autorisé par %2 : %3")
					.arg( file.canonicalPath(), station->getDisplayName(), ack);
			return false;
		}
	}
	catch(QString msg)
	{
		qDebug() << "Exception Transfert File doSendDir : " << msg;
		statutMessages += tr("Erreur réseau durant la création du dossier : %1").arg( msg);
		return false;
	}

	const QFileInfoList &files = file.entryInfoList(QDir::AllEntries | QDir::Hidden | QDir::NoSymLinks | QDir::NoDotAndDotDot);
	foreach(const QFileInfo &fInfo, files)
	{
		bool result = false;
		QDir f(fInfo.absoluteFilePath());
		if( fInfo.isSymLink())
			result = doSendLink( root, f, fInfo.symLinkTarget(), stream, station);
		else if( fInfo.isDir())
			result = doSendDir( root, f, stream, station);
		else
			result = doSendFile( root, f, stream, station);
		if( result == false)
			return false;
	}
	return true;
}

bool TransfertFile::doSendLink(const QString &root, QDir &file, QString fileTarget, ExtendedSocket *stream, Station *station)
{
	try
	{
		stream->sendLine( "makelink");
		stream->sendLine( STATIONOS + file.absolutePath().mid(root.length()));
		stream->sendLine( fileTarget.mid(root.length()));

		QString ack = stream->readLine();
		qDebug() << "makelink => " << ack;
		if( ack != "OK")
		{
			statutMessages += tr( "Création du lien %1 non autorisé par %2 : %3")
					.arg( file.canonicalPath(), station->getDisplayName(), ack);
		}
		return true;
	}
	catch(QString msg)
	{
		qDebug() << "Exception Transfert File doSendLink : " << msg;
		statutMessages += tr("Erreur réseau durant la création de lien : %1").arg(msg);
		return false;
	}
}

bool TransfertFile::doSendFile( const QString &root, QDir &file, ExtendedSocket *stream, Station *station)
{
	long long sizeAtStart = _currentSize;
start:
	FILE *f = fopen( file.absolutePath().toLocal8Bit(), "rb");
	if( f)
	{
		try
		{
			QFileInfo fInfo( file.absolutePath());
			int nbSections = (int)(fInfo.size() / TRANSFERT_BUFF_SIZE) + 1;
			int lastTransfertSize = (int)(fInfo.size() % TRANSFERT_BUFF_SIZE);
			QFile::Permissions permissions = fInfo.permissions();
			stream->sendLine( "dropfile");
			stream->sendLine( STATIONOS + file.absolutePath().mid(root.length()));
			stream->sendLine( QString::number(nbSections - 1));
			stream->sendLine( QString::number(lastTransfertSize));
			stream->sendLine( QString::number((int)permissions));

			QString ack = stream->readLine();
			qDebug() << "dropfile => " << ack;
			if( ack != "READY")
			{
				statutMessages += tr( "Transfert de %1 non autorisé par %2 : %3")
						.arg( fInfo.fileName(), station->getDisplayName(), ack);
				fclose( f);
				return false; // transfert inutile
			}
			char buff[ TRANSFERT_BUFF_SIZE];
			for( int j = 0; j < nbSections - 1; j++)
			{
				if( _exitLoopRequested) {
					stream->sendLine( tr("Transfert interrompu par l'utilisateur"));
					qDebug() << "ARRET DEMANDE PAR UTILISATEUR";
					statutMessages += tr( "Transfert de %1 interrompu par l'utilisateur").arg(fInfo.fileName());
					fclose( f);
					return false;
				}
				if( stream->state() != QAbstractSocket::ConnectedState)
				{
					qDebug() << "Perte du reseau";
					statutMessages += tr( "Transfert de %1 interrompu : perte du réseau.").arg(fInfo.fileName());
					stream->close();
					fclose( f);
					throw tr("Perte du reseau") ;
				}
				if( _pauseNeeded)
				{
					stream->sendLine("Pause");
					msleep(1000);
					QString ack = stream->readLine();
					if( ack != "pause OK")
					{
						qDebug() << "ERREUR DURANT PAUSE" << ack;
						statutMessages += tr( "Erreur durant le transfert (en pause) de %1 : %2").arg(fInfo.fileName()).arg(ack);
						fclose( f);
						return false;
					}
					j--;
					continue;
				}
				stream->sendLine("Sending");
				size_t size = fread( buff, 1, TRANSFERT_BUFF_SIZE, f);
				stream->sendBuffer( (int)size, buff);
				_currentSize += TRANSFERT_BUFF_SIZE;
				if( logDialog)
					logDialog->updateTask( this);
				QString ackMsg = stream->readLine();
				if( ackMsg == "pause") {
					_pauseAsked = true;
					if( logDialog)
						logDialog->pauseAskedTask(this);
					msleep(1000);
					stream->sendLine("pause OK");
				}
				else if( ackMsg != "partOK") {
					statutMessages += tr( "Erreur durant le transfert de %1 : %2").arg(fInfo.fileName()).arg(ackMsg);
					qDebug() << "ARRET SUR ERREUR : " << ackMsg;
					stream->close();
					fclose( f);
					return false;
				}
				else {
					if( _pauseAsked) {
						_pauseAsked = false;
						if( logDialog)
							logDialog->pauseAskedTask(this);
					}
				}
			}
			if( lastTransfertSize > 0 && stream->state() == QAbstractSocket::ConnectedState) {
				stream->sendLine("Sending");
				size_t size = fread( buff, 1, lastTransfertSize, f);
				stream->sendBuffer( (int)size, buff);
				_currentSize += lastTransfertSize;
				if( logDialog)
					logDialog->updateTask( this);

				ack = stream->readLine();
				qDebug() << " => " << ack;
				if( ack != "endOK")
				{
					statutMessages += tr( "Erreur durant le transfert de %1 : %2").arg(fInfo.fileName()).arg(ack);
					stream->close();
					fclose( f);
					return false;
				}
			}
			fclose( f);
			f = NULL;
			return true;
		}
		catch(QString msg)
		{
			if( f)
				fclose( f);
			qDebug() << "Exception Transfert File : " << msg;

			int answer = 3;
			// sur sortie décidée par utilisateur distant, ne rien tenter
			if( msg != "Transfert interrompu" && _exitLoopRequested == false) {
				emit mainWindow->continueQuestionSignal(msg);
				answer = mainWindow->answerButton;
			}
			if( answer == 3)
				throw tr("");
			else if( answer == 2 || answer == 1)
			{
				// relancer le reseau pour demander l'annulation ou la poursuite du transfert
				stream->abort();
				stream->connectToHost(_station->_address.toString(), PORTNUMBER);
				if( stream->waitForConnectedWithTimer()) {
					Station::sendInfosTo( *stream);
					if( answer == 2)
					{
						_cleanupNeeded = true;
						return false;
					}
					else if( answer == 1) {
						_currentSize = sizeAtStart;
						// refaire
						goto start;
					}
				}
			}
		}
	}
	statutMessages += tr("Impossible d'ouvrir le fichier à envoyer : %1").arg(stream->errorString());
	return false;
}

void TransfertFile::run()
{
	qDebug() << "TransfertFile RUN";
	qDebug() << "starting sending files " << _filesList;

	if( logDialog)
		logDialog->addTask(this, 1);
	try
	{
		ExtendedSocket _socket;
		_socket.connectToHost(_station->_address.toString(), PORTNUMBER);
		if( _socket.waitForConnectedWithTimer())
		{
			Station::sendInfosTo(_socket);

			bool result = false;
			foreach( const QString &fileString, _filesList)
			{
				if( _exitLoopRequested)
					break;
				QFileInfo fileToSend( fileString);
				QDir dDir(fileToSend.absoluteFilePath());
				QString root( fileToSend.dir().absolutePath());
				if( root.length() > 0 && root.at(root.length() - 1) != '/')
					root.append('/');
				if( fileToSend.isDir())
					result = doSendDir( root, dDir, &_socket, _station);
				else
					result = doSendFile( root, dDir, &_socket, _station);
				if( result == false)
					break;
			}
			if( result == true)
			{
				_socket.sendLine( "openreceivebox()");
				if( _filesList.size() == 1)
					statutMessages += tr("Envoi de %1 terminé avec succès.").arg(_filesList.first());
				else
					statutMessages += tr("Envoi de %1 fichiers terminé avec succès.").arg(_filesList.length());
			}
			else if( result == false && _cleanupNeeded)
			{
				foreach( const QString &fileString, _filesList)
				{
					QFileInfo fileToSend( fileString);
					QDir dDir(fileToSend.absoluteFilePath());
					QString root( fileToSend.dir().absolutePath());
					if( root.length() > 0 && root.at(root.length() - 1) != '/')
						root.append('/');
					_socket.sendLine("cleanup");
					qDebug() << "cleanup " << STATIONOS + dDir.absolutePath().mid(root.length());
					_socket.sendLine(STATIONOS + dDir.absolutePath().mid(root.length()));
				}
			}
			_socket.closeConnection();
		}
		else
			throw QString( "Echec de la connexion à " + _station->getDisplayName());
	}
	catch(QString msg) {
		if( !msg.isEmpty()) {
			statutMessages += tr("Erreur durant le transfert : ") + msg;
		}
	}
	_exitLoopPerformed = true;
	if( logDialog)
		logDialog->removeTask(this);
	qDebug() << "TransfertFile END";
}

long long TransfertFile::dirSize(const QString &str)
{
	long long sizex = 0;
	QFileInfo str_info(str);
	if (str_info.isDir())
	{
		QDir dir(str);
		dir.setFilter(QDir::Files | QDir::Dirs | QDir::Hidden | QDir::NoSymLinks | QDir::NoDotAndDotDot);
		QFileInfoList list = dir.entryInfoList();
		foreach(QFileInfo fileInfo, list)
		{
			if ((fileInfo.fileName() != ".") && (fileInfo.fileName() != ".."))
			{
				if(fileInfo.isDir())
					sizex += dirSize(fileInfo.filePath());
				else
					sizex += fileInfo.size();
			}
		}
	}
	else
		sizex = str_info.size();
	return sizex;
}


bool SendFileAction::checkActionValidity()
{
	Station *s = Station::findStation( _stationName);
	if( s == NULL)
		return false;
	foreach(const QString &file, _actionParametres)
	{
		QFile f(file);
		if( f.exists() == false)
			return false;
	}
	return true;
}

void SendFileAction::performAction()
{
	Station *s = Station::findStation( _stationName);
	if( s)
		s->addTransfertTask( _actionParametres);
}

bool SendClipboardAction::checkActionValidity()
{
	Station *s = Station::findStation( _stationName);
	if( s == NULL || _actionParametres.first().isEmpty())
		return false;
	return true;
}

void SendClipboardAction::performAction()
{
	Station *s = Station::findStation( _stationName);
	if( s)
		sendClipboardToStation(_actionParametres.first(), s);
}

bool GetClipboardAction::checkActionValidity()
{
	Station *s = Station::findStation( _stationName);
	if( s == NULL)
		return false;
	return true;
}

void GetClipboardAction::performAction()
{
	Station *s = Station::findStation( _stationName);
	if( s)
		getClipboardFromStation(s);
}


bool DownloadFilesAction::checkActionValidity()
{
	Station *s = Station::findStation( _stationName);
	if( s == NULL)
		return false;
	return true;
}

void DownloadFilesAction::performAction()
{
	Station *selectedStation = Station::findStation( _stationName);
	if( selectedStation == NULL)
		return;
	try
	{
		ExtendedSocket _socket;
		qDebug() << selectedStation->getDisplayName();
		_socket.connectToHost(selectedStation->_address.toString(), PORTNUMBER);
		if( _socket.waitForConnectedWithTimer())
		{
			Station::sendInfosTo(_socket);
			_socket.sendLine( "sendmefiles");
			_socket.sendLine( QString::number( _actionParametres.size()));
			foreach( const QString &file, _actionParametres)
				_socket.sendLine( file);
			_socket.closeConnection();
		}
	}
	catch(QString msg)
	{
		qDebug() << "Erreur DownloadFilesAction : " << msg;
	}
}


ExtendedSocket::ExtendedSocket() : _socket(new ipstream()), _abortRequested(false)
{
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &ExtendedSocket::abortRequested, Qt::DirectConnection);
}

ExtendedSocket::ExtendedSocket(ipstream *socket) : _socket(socket), _abortRequested(false)
{
	connect( &notificationCenter, &NotificationCenter::onApplicationQuit, this, &ExtendedSocket::abortRequested, Qt::DirectConnection);
}

ExtendedSocket::~ExtendedSocket()
{
	delete _socket;
}

void eventFct(iobase* sender, int code) {
	qDebug() << "STATUS : " << sender << code;
}

void ExtendedSocket::connectToHost(QString stationAddress, int portNumber)
{
	try {
//		_socket->set_onstatus(eventFct);
		_socket->set_host(stationAddress.toLatin1().data());
		_socket->set_port(portNumber);
		_socket->open();
	} catch( estream *error) {
		throw( QString(error->get_message()));
	}
}

void ExtendedSocket::disconnectFromHost()
{
	_socket->cancel();
}

QString ExtendedSocket::peerName()
{
	string ipaddTxt = _socket->get_host();
	return QString(ipaddTxt);
}

QHostAddress ExtendedSocket::peerAddress()
{
	ipaddress ipadd = _socket->get_ip();
	QHostAddress addr( (ipadd.data[0] << 24) + (ipadd.data[1] << 16) + (ipadd.data[2] << 8) + ipadd.data[3]);
	return addr;
}

bool ExtendedSocket::error()
{
	return false;
}

QString ExtendedSocket::errorString()
{
	return "TODO";
}

void ExtendedSocket::abortRequested()
{
	qDebug() << "ExtendedSocket ABORT Requested" << this;
	_abortRequested = true;
	_socket->cancel();
}

QAbstractSocket::SocketState ExtendedSocket::state()
{
	int status = _socket->get_status();
	if( status == IO_RESOLVING) return QAbstractSocket::HostLookupState;
	//if( status == IO_RESOLVED) return QAbstractSocket::;
	if( status == IO_CONNECTING) return QAbstractSocket::ConnectingState;
	if( status == IO_CONNECTED || status == IO_OPENED || status == IO_READING || status == IO_WRITING)
		return QAbstractSocket::ConnectedState;
	if( status == IO_CLOSING)  return QAbstractSocket::ClosingState;
	return QAbstractSocket::UnconnectedState;
}

QByteArray ExtendedSocket::readBuffer() throw(QString)
{
	try
	{
		QElapsedTimer timer;
		timer.start();
		if( _abortRequested)
			throw tr("annulation demandée");
		char stream[ sizeof( quint32)];
		int nbBytes = _socket->read( &stream, sizeof( quint32));
		if( nbBytes == sizeof( quint32))
		{
			if( _abortRequested)
				throw tr("annulation demandée");
			unsigned int blockSize = ntohl(*(unsigned int *)&stream[0]);
			qDebug() << blockSize;
			char *buff = new char[blockSize];
			unsigned int sizeMsg = _socket->read(buff, blockSize);
			if( sizeMsg == blockSize) {
				QByteArray buffer(buff, blockSize);
				delete[] buff;
				qDebug() << "<< " << buffer.left(100);
				return buffer;
			}
			delete[] buff;
		}
		throw tr("Erreur de lecture");
	} catch( estream *error) {
		throw(error ? QString(error->get_message()) : QString("NULL"));
	}
}

QString ExtendedSocket::readLine() throw(QString)
{
	return QString::fromUtf8(readBuffer());
}

void ExtendedSocket::sendLine(const QString &line) throw(QString)
{
	qDebug() << ">> " << line;
	QByteArray buffer = line.toUtf8();
	sendBuffer(buffer.length(), buffer.data());
}

void ExtendedSocket::sendBuffer(int size, char *buffer) throw(QString)
{
	quint32 len = size;
	_socket->put( ((char*)&len)[3]);
	_socket->put( ((char*)&len)[2]);
	_socket->put( ((char*)&len)[1]);
	_socket->put( ((char*)&len)[0]);
	_socket->write( buffer, size);
	_socket->flush();
}

bool ExtendedSocket::waitForConnectedWithTimer()
{
	return _socket->get_active();
}

void ExtendedSocket::abort()
{
	_socket->cancel();
	_abortRequested = true;
	return;
}

void ExtendedSocket::closeConnection()
{
	try {
		sendLine("close()");
		_socket->close();
		qDebug() << "waitForDisconnected : " << this;
		QElapsedTimer timer;
		timer.start();
		while( state() != QAbstractSocket::UnconnectedState)
		{
			QThread::msleep(30);
			//QApplication::processEvents(QEventLoop::AllEvents/* ExcludeUserInputEvents*/, 30);
			qDebug() << "WAIT for disconnection" << this;
			if( timer.elapsed() > TIMEOUT)
			{
				abort();
				qDebug() << QStringLiteral("TIMEOUT deconnexion.");
				throw( tr("Time out à la deconnexion"));
				return;
			}
			if( _abortRequested) {
				abort();
				throw tr("annulation demandée");
			}
		}
		qDebug() << "end disconnect" << this;
	}
	catch( QString ) {
	}
	catch( estream *) {
	}
}



QString LicenseManager::licenseKey = "", LicenseManager::licenseValue = "";
QDateTime LicenseManager::licenseDate;
QDateTime LicenseManager::firstRunDate;
bool LicenseManager::isLoaded = false;

int LicenseManager::testLicenseLevel( const QString &key, const QString &value)
{
	QString keyAlgo1 = QCryptographicHash::hash(key.toLocal8Bit(), QCryptographicHash::Md5).toHex().right(24).toUpper();
	qDebug() << "testLicenseLevel : " ;//<< key << value;
	if( value.length() > 0 && keyAlgo1.compare(value.toUpper().toLocal8Bit()) == 0) {
		qDebug() << "result : 2"; return 2; }
	if( firstRunDate.daysTo( QDateTime::currentDateTime()) < 30) {
		qDebug() << "result : 1"; return 1; }
	qDebug() << "result : 0";
	return 0;
}

int LicenseManager::getLicenseLevel()
{
	if( isLoaded == false)
	{
		QFile file(QCoreApplication::applicationDirPath() + "/" + LICENCEFILEPATH);
		if(!file.open(QIODevice::ReadOnly)) {
			qDebug() << "load license " << file.fileName() << " failed : " << file.errorString();
		}
		else {
			QTextStream in(&file);
			if( !in.atEnd()) licenseKey = in.readLine();
			if( !in.atEnd()) licenseValue = in.readLine();
			if( !in.atEnd()) licenseDate = QDateTime::fromString(in.readLine());
			file.close();
			isLoaded = true;
		}
	}
	return testLicenseLevel(licenseKey, licenseValue);
}

void LicenseManager::setKeyAndValue(QString _key, QString _value)
{
	licenseKey = _key;
	licenseValue = _value;
	licenseDate = QDateTime::currentDateTime();

	QFile file(QCoreApplication::applicationDirPath() + "/" + LICENCEFILEPATH);
	if(!file.open(QIODevice::ReadWrite)) {
		qDebug() << "error write license " << file.fileName() << " : " << file.errorString();
	   return;
	}

	QString txt = QString("%1\n%2\n%3\n").arg(licenseKey, licenseValue, licenseDate.toString());
	file.write( txt.toLocal8Bit());

	file.close();
	isLoaded = true;
}

bool LicenseManager::checkForLicenseLevel(int levelRequested)
{
	if( getLicenseLevel() < levelRequested)
	{
		if( licenceMessageDialog)
			return false;
		licenceMessageDialog = new QMessageBox();
		licenceMessageDialog->setText( QObject::tr("Il vous faut à présent acheter une licence.\nVoulez-vous aller sur le site pour l'acheter ?"));
		QPushButton *declineButton = licenceMessageDialog->addButton(QObject::tr("Non merci, pas encore."), QMessageBox::ActionRole);
		QObject::connect( declineButton, &QPushButton::clicked, mainWindow, &MainWindow::closeLicenceMessageDialog);
		QPushButton *buyButton = licenceMessageDialog->addButton(QObject::tr("Acheter la licence..."), QMessageBox::ActionRole);
		QObject::connect( buyButton, &QPushButton::clicked, mainWindow, &MainWindow::gotoLicenseUrl);
		licenceMessageDialog->setIcon(QMessageBox::Question);
		if( licenceMessageDialog->exec() == QMessageBox::Ok)
		{
			if( QDesktopServices::openUrl(QUrl(LICENSEURL)) == false) {
				QMessageBox::critical(NULL, QObject::tr("JOTO"), QObject::tr("Erreur lors du renvoi sur le site.\n\nVous pouvez y accéder à cette adresse : %1.").arg(LICENSEURL));
			}
		}
		if( getLicenseLevel() < levelRequested)
			return false;
	}
	return true;
}


